﻿/**
メニュー。

一応メニューバーとポップアップメニューを考慮。
まぁポップアップしか作りませんが。
*/
module nemuxi.negui.window.menu.menu;

debug import std.stdio: wl = writefln, pl = printf;
debug(menu) void main() {}

import std.string;
import std.contracts;

import win32.windows;

import nemuxi.base;
import nemuxi.system.raii;
import nemuxi.negui.negui;
import nemuxi.negui.window.menu.popup;

///
class MenuException: NeGuiException {
	mixin MixInNeGuiException;
}

/**
メニュー基底。

メニューハンドルをラップして使用。
アイテムは相対位置ではなくIDで指定する(※今後内部フラグで切り替えるようにする可能性有)。

Note:
	メニューのユーザー領域にthisを持たせたい。
*/
abstract class Menu: Raii {
	/// GC用サブメニュー。
	protected Menu[COMMAND_ID] SubMenus;
	/// メニューハンドル。
	protected HMENU hMenu;

	/**
	History:
		1.00β11:
			使用していなかったログ機能の撤廃。
	*/
	override void Kill() {
		if(this.SubMenus) {
			scope Values=this.SubMenus.values;
			foreach(key; this.SubMenus.keys) {
				this.SubMenus.remove(key);
			}
			foreach(SubMenu; Values) {
				delete SubMenu;
			}
			this.SubMenus = null;
		}
		if(hMenu) {
			if(DestroyMenu(hMenu) != 0) {
				hMenu = hMenu.init;
			} else {
				throw new NemuxiException(this.toString());
			}
		}
	}


	
	/// コンストラクタ
	this(HMENU hMenu, bool Suicide=false)
	in {
		assert(hMenu);
	}
	body {
		super(Suicide);
		this.hMenu = hMenu;
		
		//データ設定
		
	}

	version(unittest) private void ItemCheck(ref const(MENUITEM) MenuItem, string file=__FILE__, int line=__LINE__) {
		auto s=format(" [%s(%s)]", file, line);
		assert(MenuItem.id > 0, format("<Id=%s> %s", MenuItem.id, s));
	}

	/**
	メニューアイテムの挿入。

	Params:
		MenuItem = 挿入するアイテムの情報。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。
	*/
	bool insert(ref const(MENUITEM) MenuItem)
	in {
		version(unittest) ItemCheck(MenuItem);
	}
	body {
		if(MenuItem.SubMenu) {
			SubMenus[MenuItem.id] = cast(Menu)MenuItem.SubMenu;
		}
		
		return cast(bool)InsertMenuItem(
			hMenu,
			MenuItem.info.wID,
			MF_BYCOMMAND,
			MenuItem.ptr
		);
	}

	/**
	メニューアイテムの設定。

	Params:
		MenuItem = 設定するアイテムの情報。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。
	*/
	bool set(ref const(MENUITEM) MenuItem)
	in {
		version(unittest) ItemCheck(MenuItem);
	}
	body {
		if(MenuItem.SubMenu) {
			SubMenus[MenuItem.id] = cast(Menu)MenuItem.SubMenu;
		}
		
		return cast(bool)SetMenuItemInfo(
			hMenu,
			MenuItem.info.wID,
			MF_BYCOMMAND,
			MenuItem.ptr
		);
	}

	bool get(ref MENUITEM MenuItem) {
		return cast(bool)GetMenuItemInfo(
			hMenu,
			MenuItem.info.wID,
			MF_BYCOMMAND,
			MenuItem.ptr
		);
	}
	
	/**
	メニューアイテムの削除。

	Params:
		Id = 削除するアイテムID。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。
	*/
	const bool del(COMMAND_ID Id) {
		return cast(bool)DeleteMenu (
			hMenu,
			MF_BYCOMMAND,
			Id
		);
	}

	/// メニューハンドル。
	final const HMENU opCall() {
		return hMenu;
	}

	bool insertText(COMMAND_ID Id, Text text, bool Enable=true)
	in {
		assert(Id != COMMAND_ID.init);
	}
	body {
		MENUITEM MenuItem;
		MenuItem.sizeInit();
		with(MenuItem) {
			mask     = MIIM.ID | MIIM.TYPE | MIIM.STATE;
			id       = Id;
			type     = MFT.STRING;
			state    = Enable ? MFS.ENABLED: MFS.DISABLED;
		}
		MenuItem.text    = text.ptr;
		
		return insert(MenuItem);
	}
	alias insertText insertString;
	bool insertSeparator(COMMAND_ID Id=COMMAND_ID.max) {
		MENUITEM MenuItem;
		MenuItem.sizeInit();
		with(MenuItem) {
			mask     = MIIM.ID | MIIM.TYPE;
			id       = Id;
			type     = MFT.SEPARATOR;
		}
		return insert(MenuItem);
	}
	/**
	History:
		1.00β11:
			事前条件のSubMenuの判定方法を変更。
	*/
	bool insertSubMenu(COMMAND_ID Id, Text text, Menu SubMenu, bool Enable=true)
	in {
		if(SubMenu) {
			assert(isMenu(SubMenu()));
		}
	}
	body {
		MENUITEM MenuItem;
		MenuItem.sizeInit();
		with(MenuItem) {
			mask     = MIIM.ID | MIIM.TYPE | MIIM.STATE | MIIM.SUBMENU;
			id       = Id;
			type     = MFT.STRING;
			state    = Enable ? MFS.ENABLED: MFS.DISABLED;
		}
		MenuItem.text    = text.ptr;
		
		MenuItem.subMenu = SubMenu;

		return insert(MenuItem);
	}

	/**
	ID取得。

	Params:
		Position = メニューアイテムの位置。

	Return:
		PositionにあるメニューアイテムのID。

	Exception:
		Positionからメニューアイテムを取得できない場合にNemuxiExceptionを投げる。
	*/
	COMMAND_ID posToID(size_t Position) {
		MENUITEMINFO MenuItemInfo=void;
		MenuItemInfo.cbSize = MenuItemInfo.sizeof;
		MenuItemInfo.fMask = MIIM_ID;
		if(GetMenuItemInfo(hMenu, Position, MF_BYPOSITION, &MenuItemInfo)) {
			return cast(COMMAND_ID)MenuItemInfo.wID;
		}
		throw new MenuException(Text("めにゅー"));
	}

	/// 意味ね～。
	Menu getSubMenu(COMMAND_ID Id) {
		return SubMenus[Id];
	}

	/**
	ラジオチェック設定。

	Params:
		StartID = グループの先頭。
		EndID = グループの最後。
		TargetID = チェックを入れる項目。

	Exception:
		TargetIDがグループに無ければNemuxiException。

	Return:
		成功すればtrue、失敗すればfalse。
	*/
	bool radio(COMMAND_ID StartID, COMMAND_ID EndID, COMMAND_ID TargetID)
	in {
		assert(StartID);
		assert(EndID);
		assert(TargetID);
	}
	body {
		if(TargetID < StartID || EndID < TargetID) {
			throw new MenuException(Text("範囲外"));
		}
		return cast(bool)CheckMenuRadioItem(hMenu, StartID, EndID, TargetID, MF_BYCOMMAND);
	}
	/**
	指定IDのチェック状態を調べる。

	Params:
		Id = 調べたいID。

	Return:
		チェックしていればtrue、していなければfalse。

	Exception:
		メニュー項目を取得できなければNemuxiException。
	*/
	bool check(COMMAND_ID Id)
	in {
		assert(Id);
	}
	body {
		MENUITEM MenuItem;
		MenuItem.sizeInit();
		MenuItem.mask = MENUITEM.MIIM.ID | MENUITEM.MIIM.STATE;
		MenuItem.id = Id;

		if(this.get(MenuItem)) {
			return (MenuItem.state & MENUITEM.MFS.CHECKED) == MENUITEM.MFS.CHECKED;
		} else {
			throw new MenuException(Text("チェックマーク取得失敗"));
		}
	}
	/**
	指定IDのチェック状態を設定。

	Params:
		Id = 調べたいID。

		Check = ON/OFF。

	Exception:
		メニュー項目を取得・設定できなければNemuxiException。
	*/
	void check(COMMAND_ID Id, bool Check)
	in {
		assert(Id);
	}
	body {
		MENUITEM MenuItem;
		MenuItem.sizeInit();
		MenuItem.mask = MENUITEM.MIIM.ID | MENUITEM.MIIM.STATE;
		MenuItem.id = Id;

		if(this.get(MenuItem)) {
			if(Check) {
				MenuItem.state = MenuItem.state | MENUITEM.MFS.CHECKED;
			} else {
				MenuItem.state = MenuItem.state & ~MENUITEM.MFS.CHECKED;
			}
			if(!this.set(MenuItem)) {
				throw new MenuException(Text("チェックマーク設定失敗"));
			}
		} else {
			throw new MenuException(Text("メニュー項目取得失敗"));
		}
	}

	static bool isMenu(HANDLE Handle) {
		return cast(bool)IsMenu(Handle);
	}

	MENU_FLAG enable(COMMAND_ID Id, MENU_FLAG MenuFlags) {
		auto Old=EnableMenuItem(hMenu, Id, MenuFlags | MF_BYCOMMAND);
		enforce(Old != 0xFFFFFFFF, new MenuException(Text("ID(%08x)見つからず", Id)));
		return cast(MENU_FLAG)Old;
	}
}

struct MENU {
	MENUINFO MenuInfo;
	mixin(SMixInStructHiddenOriginal!(MENUINFO)(q{MenuInfo}));
}

/// メニューアイテム
struct MENUITEM {
	invariant() {
		if(info.hSubMenu) assert(SubMenu);
		else assert(!SubMenu);
	}
	/// メニューアイテムの実態だけど通常は構造体メソッドを使用。
	MENUITEMINFO info;
	mixin(SMixInStructHiddenOriginal!(MENUITEMINFO)(q{info}));
	private Menu SubMenu;

	/// 初期化サイズ。
	void initialize() {
		info.cbSize = MENUITEMINFO.sizeof;
	}
	alias initialize sizeInit;

	/// どのメンバ変数を利用するかを指定します。
	static enum MASK {
		CHECKMARKS = MIIM_CHECKMARKS, /// hbmpChecked、hbmpUnchecked メンバが有効になります。
		DATA       = MIIM_DATA      , /// dwItemData メンバが有効になります。 
		ID         = MIIM_ID        , /// wID メンバが有効になります。 
		STATE      = MIIM_STATE     , /// fState メンバが有効になります。 
		SUBMENU    = MIIM_SUBMENU   , /// hSubMenu メンバが有効になります。 
		TYPE       = MIIM_TYPE      , /// fType、dwTypeData メンバが有効になります。 
	}
	alias MASK MIIM;
	mixin(StructGetSet!(MASK)("mask", q{info.fMask}));

	/// メニュー項目のタイプを、以下の定数を組み合わせて指定します。
	static enum TYPE {
		BITMAP       = MFT_BITMAP       , /// ビットマップを使用してメニューを表示します。dwTypeData メンバの下位ワードでビットマップ ハンドルを指定します。cch メンバは無視されます。 
		MENUBARBREAK = MFT_MENUBARBREAK , /// メニュー項目を、新しい行に配置します。ドロップダウン メニュー、サブメニュー、ショートカット メニューの場合は行間に線が引かれます。 
		MENUBREAK    = MFT_MENUBREAK    , /// メニュー項目を、新しい行に配置します。ドロップダウン メニュー、サブメニュー、ショートカット メニューのときは、行間に線が引かれません。 
		OWNERDRAW    = MFT_OWNERDRAW    , /// メニューの描画をオーナーウィンドウに任せます。dwTypeData メンバはアプリケーション定義の32ビット値をもちます。 
		RADIOCHECK   = MFT_RADIOCHECK   , /// hbmpChecked メンバが NULL のとき、ラジオボタンのチェックマークをメニュー項目のチェックマークにします。 
		RIGHTJUSTIFY = MFT_RIGHTJUSTIFY , /// 以降のメニュー項目を右揃えで表示します。メニュー バーにのみ有効です。 
		RIGHTORDER   = MFT_RIGHTORDER   , /// 右から左へメニュー項目を表示します。アラビア語やヘブライ語を表示するために使用します。（Windows95、WindowsNT5.0以降） 
		SEPARATOR    = MFT_SEPARATOR    , /// メニュー項目がセパレータであることを示します。dwTypeData メンバ、cch メンバは無視されます。 
		STRING       = MFT_STRING       , /// 文字列を使ってメニュー項目を表示します。dwTypeData メンバには NULL で終わる文字列バッファへのポインタを指定します。cch メンバには文字列のバイト数を指定します。 
	}
	alias TYPE MFT;
	mixin(StructGetSet!(TYPE)("type", q{info.fType}));

	/// メニュー項目の状態を、以下の定数を組み合わせて指定します。
	static enum STATE {
		CHECKED   = MFS_CHECKED   , /// チェック マークを表示します。 
		DEFAULT   = MFS_DEFAULT   , /// デフォルトのメニュー項目に設定します。メニュー項目は太字で表示されます。 
		DISABLED  = MFS_DISABLED  , /// メニュー項目を選択不可にします。淡色表示にはなりません。 
		ENABLED   = MFS_ENABLED   , /// メニュー項目を有効にします。 
		GRAYED    = MFS_GRAYED    , /// メニュー項目を選択不可にし、淡色表示します。 
		HILITE    = MFS_HILITE    , /// メニュー項目を強調表示します。 
		UNCHECKED = MFS_UNCHECKED , /// チェック マークを解除します。 
		UNHILITE  = MFS_UNHILITE  , /// 強調表示を解除します。 
	}
	alias STATE MFS;
	mixin(StructGetSet!(STATE)("state", q{info.fState}));

	mixin(StructGetSet!(COMMAND_ID)("id", q{info.wID}));
	

	/// サブメニューを所持しているか。
	bool isSubMenu() {
		return cast(bool)info.hSubMenu;
	}
	/**
	サブメニュー取得・設定。

	Params:
		SubMenu = サブメニュー。

	Return:
		自殺しないPopUpメニュー。

	Exception:
		取得時にメニューが無効の場合にNemuxiExceptionを投げる。
	*/
	void subMenu(Menu SubMenu) {
		if(SubMenu) {
			info.hSubMenu = SubMenu.hMenu;
			this.SubMenu  = SubMenu;
			assert(info.fMask && MIIM_SUBMENU);
		} else {
			info.hSubMenu = null;
			this.SubMenu  = null;
		}
	}

	/+
	/// 項目のデータ。
	void* itemData() {
		return cast(void*)info.dwItemData;
	}
	/// ditto
	void itemData(void* ItemData) {
		info.dwItemData = cast(DWORD)ItemData;
	}
	+/
	mixin(StructGetSet!(void*)("itemData", q{info.dwItemData}));
	

	alias text typeData;
	
	/// タイプデータ。
	/// 大抵は文字列。
	wchar* text() {
		return info.dwTypeData;
	}
	/// ditto
	void text(wchar* TypeData) {
		info.dwTypeData = TypeData;
	}
	/// ditto
	wchar* text(Text TypeData) {
		return info.dwTypeData = TypeData.ptr;
	}
	
	mixin(StructGetSet!(size_t)("textLength", q{info.cch}));
}




