﻿/**
コモンダイアログボックス。

気が付けば全部クラス。
Windowのコモンダイアログを必要分だけサポート。


History:
	1.00β19:
		[S] システムダイアログの基底クラスを統一。
*/
module nemuxi.negui.window.dialog.system;

debug import std.stdio: wl = writefln, pl = printf;
debug(system) void main() {}

version(unittest) import std.string;
import std.contracts;

import win32.windows;
import win32.shlobj;

import nemuxi.base;
import nemuxi.system.raii;
public import nemuxi.negui.draw.color;
public import nemuxi.negui.draw.canvas;
public import nemuxi.negui.draw.font;
public import nemuxi.negui.window.window;
public import nemuxi.negui.window.newindow;
import nemuxi.file.file;
import nemuxi.file.folder;
import nemuxi.dll.dll;
import nemuxi.utility.meta.memberproperty;

///
class SystemDialogException: WindowException {
	mixin MixInNeGuiException;
}

template MixInSystemDialog() {
	override this() {
		super();
	}
}

interface IOKCancelDialog {
	bool select();
}

interface ITitleText {
	const Text text();
	void text(in Text);
}

abstract class SystemDialog {
	private INT ErrCode;
	final INT errCode() {
		return ErrCode;
	}
}

private enum {
	MB_APPLIMODAL  = 0,
	MB_SYSTEMMODAL = 0x1000,
	MB_TASKMODAL   = 0x2000,
}

class MessageDialog: SystemDialog, ITitleText {
	enum BUTTON {
		OK                  = MB_OK,                /// ［OK］
		OK_CANCEL           = MB_OKCANCEL,          /// ［OK］［キャンセル］
		ABORT_RETRY_IGNORE  = MB_ABORTRETRYIGNORE,  /// ［中止］［再試行］［無視］
		YES_NO_CANCEL       = MB_YESNOCANCEL,       /// ［はい］［いいえ］［キャンセル］
		YES_NO              = MB_YESNO,             /// ［はい］［いいえ］
		RETRY_CANCEL        = MB_RETRYCANCEL,       /// ［再試行］［キャンセル］
		CANCEL_TRY_CONTINUE = MB_CANCELTRYCONTINUE, /// Windows 2000/XP： ［キャンセル］［もう一度］［継続］
	}
	enum ICON {
		NONE        = 0,
		STOP        = MB_ICONSTOP,        /// 停止のアイコン
		QUESTION    = MB_ICONQUESTION,    /// 疑問符のアイコン
		EXCLAMATION = MB_ICONEXCLAMATION, /// 感嘆符のアイコン
		INFORMATION = MB_ICONINFORMATION, /// 吹き出しに「i」のアイコン 
	}
	enum DEFAULT {
		FIRST  = MB_DEFBUTTON1, /// 最初のボタンをデフォルトにする
		SECOND = MB_DEFBUTTON2, /// 2番目のボタンをデフォルトにする
		THIRD  = MB_DEFBUTTON3, /// 3番目のボタンをデフォルトにする
		FOURTH = MB_DEFBUTTON4, /// 4番目のボタンをデフォルトにする 
	}
	enum MODAL {
		APPLICATION = MB_APPLIMODAL,  /// メッセージボックスの親ウィンドウ。以外のウィンドウの操作は可能
		SYSTEM      = MB_SYSTEMMODAL, /// システムモーダルな状態
		TASK        = MB_TASKMODAL,   /// タスクモーダル
	}
	enum RESULT {
		FAIL   = 0,        /// 失敗
		ABORT  = IDABORT,  /// [中止]ボタンが押された
		CANCEL = IDCANCEL, /// [取消]ボタンが押された
		IGNORE = IDIGNORE, /// [無視]ボタンが押された
		NO     = IDNO,     /// [いいえ]ボタンが押された
		OK     = IDOK,     /// [OK]ボタンが押された
		RETRY  = IDRETRY,  /// [再試行]ボタンが押された
		YES    = IDYES,    /// [はい]ボタンがおされた
	}
	
	protected Text Title;
	mixin(ClassGetSet!(Text)("text", q{Title}, true));
	protected Text Message;
	mixin(ClassGetSet!(Text)("message", q{Message}, false));

	protected BUTTON Button;
	mixin(ClassGetSet!(BUTTON)("button", q{Button}, false));
	protected bool Help;
	mixin(ClassGetSet!(bool)("help", q{Help}, false));
	protected ICON Icon;
	mixin(ClassGetSet!(ICON)("icon", q{Icon}, false));
	protected DEFAULT Default;
	mixin(ClassGetSet!(DEFAULT)("def", q{Default}, false));

	protected NeWindow Owner;
	mixin(ClassGetSet!(NeWindow)("ownerWindow", q{Owner}, false));
	
	protected bool Forground;
	mixin(ClassGetSet!(bool)("forground", q{Forground}, false));
	protected bool TopMost;
	mixin(ClassGetSet!(bool)("topMost", q{TopMost}, false));
	protected MODAL Modal;
	mixin(ClassGetSet!(MODAL)("modal", q{Modal}, false));

	RESULT show() {
		DWORD Stateus;
		Stateus |= Help      ? MB_HELP: 0;
		Stateus |= Forground ? MB_SETFOREGROUND: 0;
		Stateus |= TopMost   ? MB_TOPMOST: 0;
		
		return cast(RESULT)MessageBox(
			Owner ? Owner(): null,
			Message.ptr,
			Title.ptr,
			Stateus | Button | Icon | Default | Modal
		);
	}
	static {
		RESULT show(Text Message) {
			auto md=new MessageDialog();
			md.message = Message;
			md.modal = MODAL.TASK;
			md.forground = true;
			md.topMost = true;
			return md.show();
		}
	}
}


/**
荷物持ちすぎ。
*/
abstract class FileDialog: SystemDialog, IOKCancelDialog, ITitleText, IInitialize {
	version(unittest) invariant() {
		if(Called && Initialized) {
			assert(cast(wchar*)OpenFileName.lpstrFilter    == Filter.ptr, format("%0.8x == %0.8x", OpenFileName.lpstrFilter, Filter.ptr));
			assert(cast(wchar*)OpenFileName.lpstrFile      == FileAddress.ptr, format("%0.8x == %0.8x", OpenFileName.lpstrFile, FileAddress.ptr));
			assert(cast(wchar*)OpenFileName.lpstrFileTitle == FileTitle.ptr);
		} else if(!Called) {
			Called=true;
		} else {
			assert(Initialized);
		}
	}
	private OPENFILENAME OpenFileName;

	this() {
		this.initialize();
	}
	version(unittest) {
		bool Initialized=false;
		bool Called=false;
	}
	override void initialize() {
		OpenFileName.lStructSize   = OpenFileName.sizeof;
		
		OpenFileName.lpstrFilter   = Filter.ptr;

		OpenFileName.nMaxFile      = FileAddress.length;
		OpenFileName.nMaxFileTitle = FileTitle.length;

		OpenFileName.lpstrFile     = FileAddress.ptr;
		OpenFileName.lpstrFileTitle= FileTitle.ptr;

		FileAddress[0] = FileTitle[0] = 0;
		delete Filter;
		version(unittest) Initialized = true;
	}

	void ownerWindow(NeWindow window) {
		OpenFileName.hwndOwner = window ? window(): null;
	}
	NeWindow ownerWindow() {
		return OpenFileName.hwndOwner
			? cast(NeWindow)NeGui.getGuiObject(OpenFileName.hwndOwner)
			: null
		;
	}

	private wchar[] Filter;
	
	struct FILTER {
		Text ShowName;
		Text[] Filters;

		static ref FILTER opCall(Text ShowName, Text[] Filters) {

			enforce(ShowName.length,new SystemDialogException(Text("ShowName.length == 0, %s", ShowName.text)));
			enforce(Filters.length,new SystemDialogException(Text("Filters.length == 0")));
			foreach(i, Filter; Filters) {
				enforce(Filter.length,new SystemDialogException(Text("Filter.length == 0, %s", Filter.text)));
			}
			
			auto Filter=new FILTER;
			
			Filter.ShowName = ShowName;
			Filter.Filters  = Filters;

			return *Filter;
		}
	}

	FILTER[] filter() {
		enforce(Filter.length, new SystemDialogException(Text("フィルタ未設定")));
		
		FILTER[] FilterList;

		bool ShowName=true;
		
		FILTER TempFilter;
		
		for(auto i=0, j=0; i < Filter.length-2; i++) {
			if(!Filter[i]) {
				size_t LastIndex=i+1;

				for(; LastIndex < Filter.length-1; LastIndex++) {
					if(!Filter[LastIndex]) {
						break;
					}
				}
				
				TempFilter.ShowName = Filter[j..i];
				TempFilter.Filters  = Text(Filter[i+1..LastIndex]).split(';');

				i = j = LastIndex+1;

				FilterList ~= TempFilter;
			}
		}
		
		return FilterList;
	}
	void filter(in FILTER[] FilterList) {
		size_t TextLength;
		foreach(Value; FilterList) {
			TextLength += Value.ShowName.length + 1;
			foreach(filter; Value.Filters) {
				TextLength += filter.length + 1;
			}
			TextLength++;
		}
		
		Filter = new wchar[++TextLength];
		OpenFileName.lpstrFilter = Filter.ptr;

		for(auto i=0, j=0; j != TextLength-1; i++) {
			Filter[j..(j + FilterList[i].ShowName.length)] = FilterList[i].ShowName.text;
			j += FilterList[i].ShowName.length;
			Filter[j++] = 0;
			
			foreach(n, filter; FilterList[i].Filters) {
				Filter[j..j + filter.length] = filter.text;
				j += filter.length;
				if(n != filter.length) {
					Filter[j++] = ';';
				} else {
					Filter[j++] = 0;
				}
			}
			Filter[j++] = 0;
		}
		Filter[$-1] = 0;


	}
	mixin(ClassGetSet!(size_t)("filterIndex", q{OpenFileName.nFilterIndex}, false));

	private wchar[PATH.MAX_PATH] FileAddress;
	Text fileAddress() {
		return Text(OpenFileName.lpstrFile);
	}
	private wchar[PATH.MAX_NAME] FileTitle;
	Text fileTitle() {
		return Text(OpenFileName.lpstrFileTitle);
	}

	const Text currentFolder() {
		return Text(cast(wchar*)OpenFileName.lpstrInitialDir);
	}
	void currentFolder(in Text Folder) {
		OpenFileName.lpstrInitialDir = Folder.ptr;
	}

	override const Text text() {
		return Text(cast(wchar*)OpenFileName.lpstrTitle);
	}
	override void text(in Text text) {
		OpenFileName.lpstrTitle = text.ptr;
	}


	static enum FLAGS {
		ALLOWMULTISELECT     = OFN_ALLOWMULTISELECT,     /// リストボックス内で、複数のファイルを選択できるようになります。 
		CREATEPROMPT         = OFN_CREATEPROMPT,         /// ユーザーが存在しないファイルを選択しようとしたときに、確認のためのダイアログを表示します。 
		ENABLEHOOK           = OFN_ENABLEHOOK,           /// lpfnHook メンバで指定されたフック関数へのポインタを使用可能にします。 
		ENABLETEMPLATE       = OFN_ENABLETEMPLATE,       /// lpTemplateName メンバと hInstance メンバで指定されるダイアログ テンプレートを利用してダイアログボックスを作成します。 
		ENABLETEMPLATEHANDLE = OFN_ENABLETEMPLATEHANDLE, /// ロードされたダイアログ テンプレートを含むデータ ブロックを、hInstance メンバが示すことを意味します。lpTemplateName メンバは無視されます。 
		EXPLORER             = OFN_EXPLORER,             /// 作成するダイアログ ボックスをエクスプローラ風にします。 
		EXTENSIONDIFFERENT   = OFN_EXTENSIONDIFFERENT,   /// ユーザーが lpstrDefExt メンバで指定されている拡張子とは異なる拡張子を入力したことを示します。 
		FILEMUSTEXIST        = OFN_FILEMUSTEXIST,        /// ユーザーが存在しないファイルを選択できないようにします。 
		HIDEREADONLY         = OFN_HIDEREADONLY,         /// 「書き込み禁止(Read Only)」チェック ボックスを非表示にします。 
		LONGNAMES            = OFN_LONGNAMES,            /// 長いファイル名を使うようにします。 
		NOCHANGEDIR          = OFN_NOCHANGEDIR,          /// ユーザーがファイル選択のためにディレクトリを移動しても、カレントディレクトリが変更されないようにします。 
		NODEREFERENCELINKS   = OFN_NODEREFERENCELINKS,   /// ショートカット ファイル(*.lnk)が選択された場合、そのショートカット ファイル自体のパスを取得します。このフラグを指定しない場合は、ショートカットのリンク先のパスを取得することになります。 
		NOLONGNAMES          = OFN_NOLONGNAMES,          /// 短いファイル名を使うようにします。エクスプローラスタイルのダイアログ ボックスでは、このフラグは無視されます。 
		NONETWORKBUTTON      = OFN_NONETWORKBUTTON,      /// 「ネットワーク」ボタンを非表示にします。 
		NOREADONLYRETURN     = OFN_NOREADONLYRETURN,     /// 取得したファイルが読み取り専用ファイル、または読み取り専用ディレクトリ内のファイルでないことを示します。 
		NOTESTFILECREATE     = OFN_NOTESTFILECREATE,     /// ダイアログ ボックスが閉じられた後にファイルを作成するようにします。 
		NOVALIDATE           = OFN_NOVALIDATE,           /// ファイル名に無効な文字を使用できることを示します。 
		OVERWRITEPROMPT      = OFN_OVERWRITEPROMPT,      /// ユーザーが選択したファイルがすでに存在するとき、上書きの確認を行います。 
		PATHMUSTEXIST        = OFN_PATHMUSTEXIST,        /// ユーザーが無効なパスやディレクトリを入力したとき、警告メッセージを表示するようにします。 
		READONLY             = OFN_READONLY,             /// ダイアログ ボックスの初期表示で、「読み取り専用」チェック ボックスをチェックします。また、ダイアログ ボックスを閉じたときに、「読み取り専用」チェック ボックスがチェックされていることを示します。 
		//SHAREWARE            = ,            /// ネットワークの共有違反でOpenFile関数が失敗した場合でも、エラーを無視してファイル名を取得するようにします。 
		SHOWHELP             = OFN_SHOWHELP,             /// ヘルプボタン[?]を表示します。

		DEFAULT              = EXPLORER | FILEMUSTEXIST | HIDEREADONLY | LONGNAMES | PATHMUSTEXIST,
	}
	alias FLAGS OFN;
	mixin(StructGetSet!(OFN)("flags", q{OpenFileName.Flags}));

	mixin(StructGetSet!(size_t)("fileOffSet", q{OpenFileName.nFileExtension}));

	Text defExtension() {
		return Text(cast(wchar*)OpenFileName.lpstrDefExt);
	}
	void defExtension(Text text) {
		OpenFileName.lpstrDefExt = text.ptr;
	}

	/+
	private INT ErrCode=0;
	INT errCode() {
		return ErrCode;
	}
	+/


	private static {
		typedef bool SAVEOPEN;
		/**
		History:
			1.00β19:
				[P] invariant変数からenumへ変更。
		*/
		enum :SAVEOPEN {
			SAVE = true,
			OPEN = false,
		}
	}
	private bool SaveOpenFile(SAVEOPEN Save) {
		auto Dialog=(
			Save == SAVE
			? &GetSaveFileName
			: &GetOpenFileName
		);

		bool ret=Dialog(&OpenFileName)
			? true
			: false
		;
		
		if(!ret) {
			ErrCode = CommDlgExtendedError();
		} else {
			ErrCode = 0;
		}

		return ret;
	}
}

class SaveDialog: FileDialog, IOKCancelDialog {
	mixin MixInSystemDialog;
	
	override bool select() {
		return super.SaveOpenFile(SAVE);
	}
}
class OpenDialog: FileDialog, IOKCancelDialog {
	mixin MixInSystemDialog;
	
	override bool select() {
		return super.SaveOpenFile(OPEN);
	}
}

private enum {
	BIF_RETURNSANCESTORS  = 0x08,
	BIF_RETURNNONLYFSDIRS = 0x01,
}
private extern(Windows)int BrowseCallbackProc(HWND hWnd, UINT Message, LPARAM lParam, LPARAM lpData) {
	if(Message == BFFM_INITIALIZED) {
		SendMessage(hWnd, BFFM_SETSELECTION, cast(WPARAM)true, lpData);
	}
	return 0;
}

class FolderDialog: SystemDialog, IOKCancelDialog, ITitleText {
	invariant() {
		assert(BrowseInfo.pszDisplayName == FolderName.ptr);
	}
	
	private BROWSEINFO BrowseInfo;

	this() {
		BrowseInfo.pszDisplayName = FolderName.ptr;
		FolderName[0] = 0;
		BrowseInfo.lParam=cast(LPARAM)FolderAddress.ptr;
		BrowseInfo.lpfn=cast(BFFCALLBACK)&BrowseCallbackProc;
	}
	mixin(StructGetSet!(BFFCALLBACK)("callBack", q{BrowseInfo.lpfn}));
	
	void ownerWindow(NeWindow window) {
		BrowseInfo.hwndOwner = window ? window(): null;
	}
	NeWindow ownerWindow() {
		return BrowseInfo.hwndOwner
			? cast(NeWindow)NeGui.getGuiObject(BrowseInfo.hwndOwner)
			: null
		;
	}
	
	private wchar[PATH.MAX_NAME] FolderName;
	Text folderName() {
		return Text(BrowseInfo.pszDisplayName);
	}
	
	override const Text text() {
		return Text(cast(wchar*)BrowseInfo.lpszTitle);
	}
	override void text(in Text text) {
		BrowseInfo.lpszTitle = text.ptr;
	}
	static enum FLAGS {
		BROWSEFORCOMPUTER  = BIF_BROWSEFORCOMPUTER, /// コンピュータフォルダのみ選択可能
		BROWSEFORPRINTER   = BIF_BROWSEFORPRINTER,  /// プリンタフォルダのみ選択可能
		BROWSEINCLUDEFILES = BIF_BROWSEINCLUDEFILES,/// ファイルも表示する
		DONTGOBELOWDOMAIN  = BIF_DONTGOBELOWDOMAIN, /// ネットワークフォルダを含まない
		EDITBOX            = BIF_EDITBOX,           /// ダイアログボックス内にアイテム名入力用のテキストボックスを追加する
		RETURNSANCESTORS   = BIF_RETURNSANCESTORS,  /// 親ディレクトリのみ選択できるようにする
		RETURNNONLYFSDIRS  = BIF_RETURNNONLYFSDIRS, /// ディレクトリのみ選択可能
		STATUSTEXT         = BIF_STATUSTEXT,        /// ダイアログボックスにステータス表示領域を追加する
		VALIDATE           = BIF_VALIDATE,          /// 無効なアイテム名が入力されたときコールバック関数のBrowseCallbackProcを呼び出す
		NEWDIALOGSTYLE     = BIF_NEWDIALOGSTYLE,    /// 新しいスタイルのダイアログボックスを使います。サイズ変更やその他の機能が付加されています。(詳細はヘルプ参照)
		USENEWUI           = BIF_USENEWUI,          /// BIF_EDITBOX | BIF_NEWDIALOGSTYLEと同じです。

		DEFAULT            = BROWSEFORCOMPUTER | RETURNNONLYFSDIRS | DONTGOBELOWDOMAIN | RETURNSANCESTORS |RETURNNONLYFSDIRS | USENEWUI,
	}
	alias FLAGS BIF;
	mixin(StructGetSet!(FLAGS)("flags", q{BrowseInfo.ulFlags}));

	private WCHAR[PATH.MAX_PATH] FolderAddress;
	override bool select() {
		auto IDList = SHBrowseForFolder(&BrowseInfo);

		if(IDList) {
			SHGetPathFromIDList(IDList, FolderAddress.ptr);
			CoTaskMemFree(IDList);
			return true;
		} else {
			return false;
		}
	}
	Text folderAddress()
	out(r) {
		if(r.length) {
			assert(PATH.isFolderPath(r), r.toString);
		}
	}
	body {
		auto Path=Text(FolderAddress.ptr);
		if(Path.length) {
			return PATH.addFolderSep(Path);
		}
		throw new SystemDialogException(Text("選択不能フォルダ"));
	}
	void folderAddress(in Text Path) {
		auto Length=Path.text.length;
		FolderAddress[0..Length] = Path.text[0..Length];
		FolderAddress[Length] = 0;
		BrowseInfo.lParam=cast(LPARAM)FolderAddress.ptr;
	}
}
/+
debug(system) unittest {
	auto f=new FolderDialog();
	f.text=Text("さぎょうふぉるだのせんたく");
	f.folderAddress(Text(`C:\Documents and Settings\sk\Local Settings\Application Data\Microsoft\Messenger\abcdefghijklmnopqrstuvwxyz012345678976japan@hotmail.com\Sharing Folders\big_ups_note@hotmail.co.jp\`));
	with(FolderDialog.BIF)f.flags=RETURNNONLYFSDIRS|USENEWUI;
	if(f.select()) {
		wl("> ",f.folderAddress);
	}
}
+/

class IconDialog: SystemDialog, IOKCancelDialog {
	private {
		typedef extern(Windows) BOOL function(HWND, LPWSTR, DWORD, DWORD*) SHChangeIconDialogType;
		static SHChangeIconDialogType SHChangeIconDialog;
		static DLL ShellDLL;
		static this() {
			ShellDLL = new DLL(Text("shell32"));
			SHChangeIconDialog = ShellDLL.address!(SHChangeIconDialogType)(62);
		}
		static ~this() {
			delete ShellDLL;
		}
	}
	
	this() {
		IconAddress[0] = 0;
		IconIndex = 0;
	}
	private wchar[PATH.MAX_PATH] IconAddress;
	Text iconAddress() {
		return Text(IconAddress.ptr);
	}
	void iconAddress(in Text Path) {
		auto Length=Path.text.length;
		IconAddress[0..Length] = Path.text[0..Length];
		IconAddress[Length] = 0;
	}
	private DWORD IconIndex;
	mixin(StructGetSet!(DWORD)("iconIndex", q{IconIndex}));

	private NeWindow OwnerWindow;
	void ownerWindow(NeWindow window) {
		OwnerWindow = window;
	}
	NeWindow ownerWindow() {
		return OwnerWindow
			? OwnerWindow
			: null
		;
	}
	
	override bool select() {
		return cast(bool)SHChangeIconDialog(OwnerWindow(), IconAddress.ptr, IconAddress.length, &IconIndex);
	}
}


/**
なんだこれ。
*/
class FontDialog: SystemDialog, IOKCancelDialog, IInitialize {
	private CHOOSEFONT Choose;

	this() {
		this.initialize();
	}
	
	void ownerWindow(NeWindow window) {
		Choose.hwndOwner = window ? window(): null;
	}
	NeWindow ownerWindow() {
		return cast(NeWindow)NeGui.getGuiObject(Choose.hwndOwner);
	}

	Canvas canvas() {
		return Choose.hDC ? new Canvas(Choose.hDC, false): null;
	}
	void canvas(Canvas canvas) {
		Choose.hDC = canvas ? canvas(): null;
	}

	void font(Font font) {
		Choose.lpLogFont = &font.getStruct();
	}
	Font font() {
		return new Font(Choose.lpLogFont);
	}

	mixin(StructGetSet!(int)("point", q{Choose.iPointSize}));

	///
	enum FLAGS {
		SCREENFONTS          = CF_SCREENFONTS          , /// システムがサポートしているスクリーンフォントのみがリスト表示されます。
		PRINTERFONTS         = CF_PRINTERFONTS         , /// hDCメンバで指定されたデバイスコンテキスト（または情報コンテキスト）に関連付けられているプリンタがサポートするフォントのみがリスト表示されます。
		BOTH                 = CF_BOTH                 , /// システムがサポートしているスクリーンフォントとhDCメンバで指定されたデバイスコンテキスト（または情報コンテキスト）に関連付けられているプリンタがサポートするフォントがリスト表示されます。このフラグはCF_SCREENFONTSとCF_PRINTERFONTSの組み合わせです。
		SHOWHELP             = CF_SHOWHELP             , /// ヘルプボタンを表示します。hwndOwnerメンバには、ヘルプボタンが押されたときに送られるHELPMSGSTRING ("commdlg_help") 登録メッセージを受け取るウィンドウを指定しなければなりません。
		ENABLEHOOK           = CF_ENABLEHOOK           , /// lpfnHookメンバで指定されたフックプロシージャを有効にします。
		ENABLETEMPLATE       = CF_ENABLETEMPLATE       , /// hInstanceメンバとlpTemplateNameメンバで指定されたダイアログボックステンプレートをデフォルトテンプレートの代わりに使用します。
		ENABLETEMPLATEHANDLE = CF_ENABLETEMPLATEHANDLE , /// hInstanceメンバで指定されたメモリブロックに含まれるダイアログボックステンプレートを使用します。lpTemplateNameメンバは無視されます。
		INITTOLOGFONTSTRUCT  = CF_INITTOLOGFONTSTRUCT  , /// ダイアログボックスをlpLogFontメンバが指し示すLOGFONT構造体で指定されたフォントが選択された状態に初期化します。
		USESTYLE             = CF_USESTYLE             , /// フォントスタイルコンボボックスをlpszStyleメンバが指し示すバッファで指定されたスタイルデータで初期化します。ユーザーがダイアログボックスを閉じた後に、このバッファにスタイルデータが格納されます。
		EFFECTS              = CF_EFFECTS              , /// ユーザーが打ち消し線・下線・文字色を指定できるようにします。このフラグが指定されると、rgbColorsに初期の文字色を指定することができます。また、lpLogFontメンバが指し示すLOGFONT構造体のlfStrikeOutメンバとlfUnderlineメンバに初期の設定を指定することができます。ユーザーがダイアログボックスを閉じた後に、これらのメンバに選択された設定が格納されます。
		APPLY                = CF_APPLY                , /// ダイアログボックスに [適用] ボタンを表示します。フックプロシージャ中でWM_COMMANDメッセージを処理する必要があります。フックプロシージャは、ダイアログボックスにWM_CHOOSEFONT_GETLOGFONTメッセージを送信して、現在選択されているフォント情報が格納されたLOGFONT構造体を取得することができます。
		SCRIPTSONLY          = CF_SCRIPTSONLY          , /// 非OEM文字セットやシンボル文字セットを選択できるようにします。
		NOVECTORFONTS        = CF_NOVECTORFONTS        , /// ベクトルフォントを選択できないようにします。
		NOSIMULATIONS        = CF_NOSIMULATIONS        , /// GDIフォントシミュレーションを行なわないようにします。
		LIMITSIZE            = CF_LIMITSIZE            , /// nSizeMinメンバおよびnSizeMaxメンバで、フォントサイズの範囲を制限します。
		FIXEDPITCHONLY       = CF_FIXEDPITCHONLY       , /// 固定ピッチフォントのみを選択できるようにします。
		WYSIWYG              = CF_WYSIWYG              , /// ディスプレイとプリンタの両方で使用可能なフォントのみを選択できるようにします。CF_BOTHとCF_SCALABLEONLYをともに指定しなければなりません。
		FORCEFONTEXIST       = CF_FORCEFONTEXIST       , /// 存在しないフォントやスタイルをユーザーが選択しようした場合には、関数がエラーを返すようにします。
		SCALABLEONLY         = CF_SCALABLEONLY         , /// スケーリング可能なフォント（ベクトルフォント、スケーリング可能プリンタフォント、TrueTypeフォント、およびその他の技術によってスケーリング可能なフォント）のみを選択できるようにします。
		TTONLY               = CF_TTONLY               , /// TrueTypeフォントのみを選択できるようにします。
		NOFACESEL            = CF_NOFACESEL            , /// ダイアログボックスの初期状態を設定するのにLOGFONT構造体の情報を使用する場合、初期状態でフォント名のコンボボックスに何も選択されていないようにします。
		NOSTYLESEL           = CF_NOSTYLESEL           , /// ダイアログボックスの初期状態を設定するのにLOGFONT構造体の情報を使用する場合、初期状態でフォントスタイルのコンボボックスに何も選択されていないようにします。
		NOSIZESEL            = CF_NOSIZESEL            , /// ダイアログボックスの初期状態を設定するのにLOGFONT構造体の情報を使用する場合、初期状態でフォントサイズのコンボボックスに何も選択されていないようにします。
		SELECTSCRIPT         = CF_SELECTSCRIPT         , /// LOGFONT構造体のlfCharSetメンバで指定された文字セットを持つフォントのみが表示されます。文字セットコンボボックスで指定される文字セットをユーザーが変更することができないようになります。
		NOSCRIPTSEL          = CF_NOSCRIPTSEL          , /// 文字セットのコンボボックスを無効にします。ユーザーがダイアログボックスを閉じた後に、LOGFONT構造体のlfCharSetメンバには1 (DEFAULT_CHARSET) が格納されます。
		NOVERTFONTS          = CF_NOVERTFONTS          , /// 水平方向のフォントのみがリスト表示されます。
	}
	mixin(StructGetSet!(FLAGS)("flags", q{Choose.Flags}));
	
	COLOR color() {
		return COLOR(Choose.rgbColors);
	}
	void color(COLOR color) {
		Choose.rgbColors = color.code;
	}

	/// 選択されたフォントのタイプが格納されます。以下の値の組み合わせになります。
	enum TYPE {
		BOLD      = BOLD_FONTTYPE      , /// フォントは太字体のフォントです。この情報は、LOGFONT構造体のlfWeightメンバにも反映され、FW_BOLDと同等になります。
		ITALIC    = ITALIC_FONTTYPE    , /// イタリック体のフォント属性が設定されます。この情報は、LOGFONT構造体のlfItalicメンバにも反映されます。
		REGULAR   = REGULAR_FONTTYPE   , /// フォントは通常の太さのフォントです。この情報は、LOGFONT構造体のlfWeightメンバにも反映され、FW_REGULARと同等になります。
		SCREEN    = SCREEN_FONTTYPE    , /// フォントはスクリーンフォントです。
		PRINTER   = PRINTER_FONTTYPE   , /// フォントはプリンタフォントです。
		SIMULATED = SIMULATED_FONTTYPE , /// フォントはGDIによってシミュレートされます。
	}
	mixin(StructGetSet!(TYPE)("type", q{Choose.nFontType}));
	
	mixin(StructGetSet!(int)("min", q{Choose.nSizeMin}));
	mixin(StructGetSet!(int)("max", q{Choose.nSizeMax}));

	/+
	private INT ErrCode=0;
	INT errCode() {
		return ErrCode;
	}
	+/
	
	override void initialize() {
		Choose.lStructSize = Choose.sizeof;
	}
	override bool select() {
		bool ret=cast(bool)ChooseFont(&Choose);

		if(ret) {
			ErrCode = 0;
		} else {
			ErrCode = CommDlgExtendedError();
		}
		
		return ret;
	}
}

class ColorDialog: SystemDialog, IInitialize, IOKCancelDialog {
	invariant() {
		version(unittest) if(Initialized) {
			assert(Choose.lpCustColors == cast(COLORREF*)customColor.ptr);
		}
	}
	private CHOOSECOLOR Choose;
	enum {
		CUSTOMCOLOR_SIZE = 16
	}

	version(unittest) private bool Initialized=false;
	override void initialize() {
		Choose.lStructSize  = Choose.sizeof;
		Choose.lpCustColors = cast(COLORREF*)customColor.ptr;

		version(unittest) Initialized=true;
	}
	
	void ownerWindow(NeWindow window) {
		Choose.hwndOwner = window ? window(): null;
	}
	NeWindow ownerWindow() {
		return cast(NeWindow)NeGui.getGuiObject(Choose.hwndOwner);
	}

	COLOR color() {
		return COLOR(Choose.rgbResult);
	}
	void color(COLOR color) {
		Choose.rgbResult = color.code;
	}

	COLOR[CUSTOMCOLOR_SIZE] customColor;

	/// ダイアログボックスの初期化フラグを、以下の定数を組み合わせて指定します。
	enum FLAGS {
		ENABLEHOOK           = CC_ENABLEHOOK           , ///lpfnHook メンバを有効にします。 
		ENABLETEMPLATE       = CC_ENABLETEMPLATE       , ///lpTemplateName メンバと hInstance メンバで指定されるダイアログ テンプレートを利用してダイアログボックスを作成します。 
		ENABLETEMPLATEHANDLE = CC_ENABLETEMPLATEHANDLE , ///ロードされたダイアログ テンプレートを含むデータ ブロックを、hInstance メンバが示すことを意味します。lpTemplateName メンバは無視されます。 
		FULLOPEN             = CC_FULLOPEN             , ///カスタム カラー部を含む、ダイアログボックス全体を表示します。 
		PREVENTFULLOPEN      = CC_PREVENTFULLOPEN      , ///「色の作成」ボタンを無効にします。 
		RGBINIT              = CC_RGBINIT              , ///rgbResult メンバで指定した色を初期カラーとして使用します。 
		SHOWHELP             = CC_SHOWHELP             , ///ダイアログボックスにヘルプボタン「?」を表示します。 
	}
	mixin(StructGetSet!(FLAGS)("flags", q{Choose.Flags}));

	override bool select() {
		bool ret=cast(bool)ChooseColor(&Choose);

		if(ret) {
			ErrCode = 0;
		} else {
			ErrCode = CommDlgExtendedError();
		}
		
		return ret;
	}
}



