﻿/**
タイマーというか時間関係。

std.timeがあまりにもあれだったんで置き換え。


Bugs:
	StructGetじゃだめ、ClassGetにせねば。
*/
module nemuxi.negui.system.timer;

debug import std.stdio: wl = writefln, pl = printf;
debug(timer) void main() {}

import std.perf;
public import std.date;
version(unittest) import std.regexp;
import std.string;
import std.contracts;
import std.conv;
import std.math;

import win32.windows;

import nemuxi.negui.system.text;
import nemuxi.negui.system.exception;
//import nemuxi.system.nemuxierrors;
import nemuxi.utility.meta.memberproperty;

///
class DateTimeException: NeGuiException {
	mixin MixInNeGuiException;
}

debug scope class DTimer: PerformanceCounter {
	this() {
		start();
	}

	~this() {
		stop();
		wl("Timer:%s", microseconds);
	}
}

/**
History:
	1.00β19:
		新規作成。
*/
class Time {
	protected SYSTEMTIME SystemTime;
	mixin(SMixInStructHiddenOriginal!(SYSTEMTIME)(q{SystemTime}));
}

/**
日時を扱う専用クラス。

ﾈﾑぃに特化。

History:
	1.00β19:
		[S] Timeを継承。
*/
class DateTime: Time {
	invariant() {
		assert(Mask == MASK.DATE || Mask == MASK.TIME || (Mask == MASK.DATE | MASK.TIME));
		with(SystemTime) {
			if(Mask & MASK.DATE) {
				assert(1 <= wMonth     && wMonth     <= 12);
				assert(0 <= wDayOfWeek && wDayOfWeek <= 6);
				assert(1 <= wDay       && wDay       <= 31);
			}
			if(Mask & MASK.TIME) {
				assert(0 <= wHour   && wHour   <= 23);
				assert(0 <= wMinute && wMinute <= 59);
				assert(0 <= wSecond && wSecond <= 59);
			}
		}

		version(unittest) assert(find(Format, RegExp(`\[\w+\]`)) != -1, Format);

		size_t[string] hash;
		foreach(month; MonthList) {
			assert((hash[month] += 1) == 1, Text("重複？[%s:%s]", month, hash[month]).text8);
		}
		hash=null;
		foreach(week; WeekList) {
			assert((hash[week] += 1) == 1, Text("重複？[%s:%s]", week, hash[week]).text8);
		}
	}
	/+
	protected SYSTEMTIME SystemTime;
	mixin(SMixInStructHiddenOriginal!(SYSTEMTIME)(q{SystemTime}));
	+/

	private string FormatTenuki(in string FormatTarget, in string Token, in size_t Width,  in int Value) {
		if(Width) {
			//auto s=std.string.toString(Value);
			auto s=to!(string)(Value);
			if(s.length >= Width) {
				s = s[s.length - Width .. s.length];
			} else {
				//s = repeat("0", Width-1) ~ s;
				//s = Text("%0" ~ std.string.toString(Width) ~ "s", Value);
				s = Text("%0" ~ to!(string)(Width) ~ "s", Value).text8;
			}
			return replace(FormatTarget, Token, s);//sub(FormatTarget, RegToken, s);
		} else {
			//return sub(FormatTarget, RegToken, std.string.toString(Value));
			//return sub(FormatTarget, RegToken, to!(string)(Value));
			return replace(FormatTarget, Token, to!(string)(Value));//sub(FormatTarget, RegToken, s);
		}
	}
	
	/**
	拡張はとりあえず置いておく。
	*/
	protected string DateTimeFormat(in string Format, in string[] MonthList, in string[] WeekList)
	in {
		assert(Format.length);
		assert(MonthList.length == MONTH_SIZE);
		assert(WeekList.length  == WEEK_SIZE);
	}
	body {
		string Formated=Format;

		
		if(Mask & MASK.DATE) {
			// 年
			Formated = FormatTenuki(Formated, "[Y]", 0, SystemTime.wYear);
			Formated = FormatTenuki(Formated, "[YY]", 2, SystemTime.wYear);
			Formated = FormatTenuki(Formated, "[YYYY]", 4, SystemTime.wYear);

			Formated = replace(Formated, "[M]", MonthList[SystemTime.wMonth-1]);

			Formated = FormatTenuki(Formated, "[D]", 0, SystemTime.wDay);
			Formated = FormatTenuki(Formated, "[DD]", 2, SystemTime.wDay);
			
			Formated = replace(Formated, "[W]", WeekList[SystemTime.wDayOfWeek]);
		}
		if(Mask & MASK.TIME) {
			// 何も考えずに
			Formated = FormatTenuki(Formated, "[h]", 0, SystemTime.wHour);
			Formated = FormatTenuki(Formated, "[hh]", 2, SystemTime.wHour);

			Formated = FormatTenuki(Formated, "[m]", 0, SystemTime.wMinute);
			Formated = FormatTenuki(Formated, "[mm]", 2, SystemTime.wMinute);

			Formated = FormatTenuki(Formated, "[s]", 0, SystemTime.wSecond);
			Formated = FormatTenuki(Formated, "[ss]", 2, SystemTime.wSecond);

			Formated = FormatTenuki(Formated, "[ms]", 0, SystemTime.wMilliseconds);
			Formated = FormatTenuki(Formated, "[MS]", 3, SystemTime.wMilliseconds);
		}

		return Formated;
	}
	

	private void _this() {
		Mask = MASK.DATE | MASK.TIME;

		Format    = FORMAT;
		MonthList = MONTHLIST;
		WeekList  = WEEKLIST;
	}

	/***/
	this() {
		GetLocalTime(&SystemTime);
		
		_this();
	}
	debug(timer) unittest {
		new DateTime;
	}

	this(in d_time UtcTime) {
		SystemTime.wYear         = cast(WORD)yearFromTime(UtcTime);
		SystemTime.wMonth        = cast(WORD)monthFromTime(UtcTime);
		SystemTime.wDay          = cast(WORD)dateFromTime(UtcTime);
		SystemTime.wDayOfWeek    = cast(WORD)weekDay(UtcTime);
		SystemTime.wHour         = cast(WORD)hourFromTime(UtcTime);
		SystemTime.wMinute       = cast(WORD)minFromTime(UtcTime);
		SystemTime.wSecond       = cast(WORD)secFromTime(UtcTime);
		SystemTime.wMilliseconds = cast(WORD)msFromTime(UtcTime);

		_this();
	}
	this(in SYSTEMTIME* SystemTime) {
		this.SystemTime = *SystemTime;
		
		_this();
	}
	debug(timer) unittest {
		new DateTime(UTCtoLocalTime(getUTCtime()));
	}
	this(in WORD Year, in WORD Month, in WORD Day, in WORD Hour, in WORD Minute, in WORD Second, in WORD MilliSecond) {
		SystemTime.wYear         = Year;
		SystemTime.wMonth        = Month;
		SystemTime.wDay          = Day;
		SystemTime.wHour         = Hour;
		SystemTime.wMinute       = Minute;
		SystemTime.wSecond       = Second;
		SystemTime.wMilliseconds = MilliSecond;
		// 曜日取得
		ChangeWeek();
		
		_this();
	}
	/**
	曜日の取得。

	Standards:
		Zellerの公式
	See_Also:
		ウィキペディア, http://ja.wikipedia.org/wiki/%E3%83%84%E3%82%A7%E3%83%A9%E3%83%BC%E3%81%AE%E5%85%AC%E5%BC%8F
	*/
	static WORD getWeekDay(in WORD Year, in WORD Month, in WORD Day)
	in {
		assert(1 <= Month && Month <= 12);
		assert(1 <= Month && Month <= 31);
	}
	body {
		immutable YEAR  = Month > 2 ? Year:  Year - 1;
		immutable MONTH = Month > 2 ? Month: Month + 12;
		return cast(WORD)((YEAR + floor(YEAR / 4) - floor(YEAR / 100) + floor(YEAR / 400) + floor(2.6 * MONTH + 1.6) + Day) % 7);
	}
	protected void ChangeWeek() {
		SystemTime.wDayOfWeek = getWeekDay(SystemTime.wYear, SystemTime.wMonth, SystemTime.wDay);
	}

	/**
	わざわざビット演算しなくてもいいよなぁ。
	*/
	static enum MASK {
		DATE = 0b0001,
		TIME = 0b0010
	}
	protected MASK Mask;
	mixin(StructGetSet!(MASK)("mask", q{this.Mask}));

	static {
		invariant FORMAT    = "[Y]/[M]/[D]([W]) [h]:[m]:[s].[ms]";
		invariant MONTHLIST = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12"];
		invariant WEEKLIST  = ["日", "月", "火", "水", "木", "金", "土"];
	}

	protected string Format;
	/**
	<table>
		<thead>
			<tr>
				<th>変換前</th>
				<th>変換後</th>
			</tr>
		</thead>
		<tbody>
			<tr>
				<td>[Y]</td>
				<td>12345:12345, 1234:1234, 123:123, 12:12, 1:1</td>
			</tr>
			<tr>
				<td>[YY]</td>
				<td>12345:45, 1234:34, 123:23, 12:12, 1:01</td>
			</tr>
			<tr>
				<td>[YYYY]</td>
				<td>12345:2345, 1234:1234, 123:0123, 12:0012, 1:0001</td>
			</tr>
			<tr>
				<td>[M]</td>
				<td>DateTime.monthList()</td>
			</tr>
			<tr>
				<td>[D]</td>
				<td>12345:12345, 1234:1234, 123:123, 12:12, 1:1</td>
			</tr>
			<tr>
				<td>[DD]</td>
				<td>12345:45, 1234:34, 123:23, 12:12, 1:01</td>
			</tr>
			<tr>
				<td>[W]</td>
				<td>DateTime.weekList()</td>
			</tr>
			<tr>
				<td>[h]</td>
				<td>12345:12345, 1234:1234, 123:123, 12:12, 1:1</td>
			</tr>
			<tr>
				<td>[hh]</td>
				<td>12345:45, 1234:34, 123:23, 12:12, 1:01</td>
			</tr>
			<tr>
				<td>[m]</td>
				<td>12345:12345, 1234:1234, 123:123, 12:12, 1:1</td>
			</tr>
			<tr>
				<td>[mm]</td>
				<td>12345:45, 1234:34, 123:23, 12:12, 1:01</td>
			</tr>
			<tr>
				<td>[s]</td>
				<td>12345:12345, 1234:1234, 123:123, 12:12, 1:1</td>
			</tr>
			<tr>
				<td>[ss]</td>
				<td>12345:45, 1234:34, 123:23, 12:12, 1:01</td>
			</tr>
			<tr>
				<td>[ms]</td>
				<td>12345:12345, 1234:1234, 123:123, 12:12, 1:1</td>
			</tr>
			<tr>
				<td>[MS]</td>
				<td>12345:345, 1234:234, 123:123, 12:012, 1:001</td>
			</tr>
		</tbody>
	</table>
	*/
	/+
	string format() {
		return Format;
	}
	void format(string Format) {
		this.Format = Format;
	}
	+/
	mixin(StructGetSet!(string)("format", q{this.Format}));
	
	debug(timer) unittest {
		auto t=new DateTime(
			2001, 12, 31,
			23, 59, 59, 999
		);
		assert(t.toString() == "2001/12/31(月) 23:59:59.999", t.toString());

		t.year = 12345;
		t.format = "[Y]";
		assert(t.toString() == "12345");
		t.format = "[YY]";
		assert(t.toString() == "45");
		t.format = "[YYYY]";
		assert(t.toString() == "2345");
		t.format = "[MS]";
		t.ms=9;
		assert(t.toString() == "009");
		t.ms=99;
		assert(t.toString() == "099");
		t.format = "[ms]";
		t.ms=9;
		assert(t.toString() == "9");
		t.ms=99;
		assert(t.toString() == "99");

		t = new DateTime(2000,1,1,0,0,0,0);
		assert(t.week == 6);
		t.month = 2;
		assert(t.week == 2);
		t.month = 3;
		assert(t.week == 3);
	}
	static invariant MONTH_SIZE = 12;
	protected string[MONTH_SIZE] MonthList;
	string[] monthList() const {
		return cast(string[])MonthList;
	}
	void monthList(in string[] MonthList) {
		foreach(Month; MonthList) {
			if(std.string.indexOf(Month, '[') != -1) {
				throw new DateTimeException(Text("不正"), NGEC.DATETIME_MONTH_LIST);
			}
		}
		this.MonthList = MonthList;
	}
	
	static invariant WEEK_SIZE = 7;
	protected string[WEEK_SIZE] WeekList;
	string[] weekList() const {
		return cast(string[])WeekList;
	}
	void weekList(in string[] WeekList) {
		foreach(Week; WeekList) {
			if(std.string.indexOf(Week, '[') != -1) {
				throw new DateTimeException(Text("不正"), NGEC.DATETIME_WEEKLIST);
			}
		}
		this.WeekList = WeekList;
	}

	mixin(StructGet!(WORD)("year", q{SystemTime.wYear}));
	void year(in WORD Year) {
		SystemTime.wYear = Year;
		ChangeWeek();
	}
	
	mixin(StructGet!(WORD)("month", q{SystemTime.wMonth}));
	void month(WORD Month) {
		enforce(1 <= Month && Month <= 12, new DateTimeException(Text("月の値が不正(%s)", Month), NGEC.DATETIME_MONTH));
		SystemTime.wMonth = Month;
		ChangeWeek();
	}

	mixin(StructGet!(WORD)("day", q{SystemTime.wDay}));
	void day(WORD Day) {
		enforce(1 <= Day && Day <= 31, new DateTimeException(Text("日の値が不正(%s)", Day), NGEC.DATETIME_DAY));
		SystemTime.wDay = Day;
		ChangeWeek();
	}
	WORD week() const {
		return SystemTime.wDayOfWeek;
	}

	string weekName(WORD WeekIndex) {
		enforce(0 <= WeekIndex && WeekIndex <= 6, new DateTimeException(Text("週の値が不正(%s)", WeekIndex), NGEC.DATETIME_WEEK_NAME));
		return WeekList[WeekIndex];
	}
	
	mixin(StructGet!(WORD)("hour", q{SystemTime.wHour}));
	void hour(WORD Hour) {
		enforce(0 <= Hour && Hour <= 23, new DateTimeException(Text("時間の値が不正(%s)", Hour), NGEC.DATETIME_HOUR));
		SystemTime.wHour = Hour;
	}

	mixin(StructGet!(WORD)("minute", q{SystemTime.wMinute}));
	void minute(WORD Minute) {
		enforce(0 <= Minute && Minute <= 59, new DateTimeException(Text("分の値が不正(%s)", Minute), NGEC.DATETIME_MINUTE));
		SystemTime.wMinute = Minute;
	}
	
	mixin(StructGet!(WORD)("second", q{SystemTime.wSecond}));
	void second(WORD Second) {
		enforce(0 <= Second && Second <= 59, new DateTimeException(Text("秒の値が不正(%s)", Second), NGEC.DATETIME_SECOND));
		SystemTime.wSecond = Second;
	}

	mixin(StructGetSet!(WORD)("ms", q{SystemTime.wMilliseconds}));
	

	bool opEqual(Object obj) const {
		if(auto date=cast(DateTime)obj) {
			if(Mask == date.Mask) {
				return SystemTime == date.SystemTime;
			}
		}
		
		return false;
	}

	override string toString() {
		return DateTimeFormat(Format, MonthList, WeekList);
	}
}
/+
package class ThrowableTime: DateTime {
	override this(in d_time utc) {
		super(utc);
		Format    = "[YYYY]/[M]/[DD] [hh]:[mm]:[ss].[MS]";
		MonthList = ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"];
	}
}
+/
