﻿/**

*/
module nemuxi.negui.draw.font;

debug import std.stdio: wl = writefln, pl = printf;

debug(font) void main() {}

import std.contracts;

import win32.core;

import nemuxi.base;
import nemuxi.system.raii;
import nemuxi.negui.draw.draw;


///
class FontException: DrawException {
	mixin MixInNeGuiException;
}


/**
フォントラッパークラス。

*/
class Font: Draw {
	/// フォントハンドル。
	protected alias Handle hFont;
	mixin KillResource!("フォント", hFont, DeleteObject, 0);

	protected mixin GetImportObject!("フォント", hFont, LOGFONT, CreateFontIndirect);
	ref LOGFONT getStruct() {
		return GetStruct(hFont);
	}
	/**
	History:
		1.00β17:
			新規追加。
	*/
	Font dup() {
		return new Font(&GetStruct(hFont));
	}
	
	///
	override this(HFONT hFont, bool Suicide) {
		super(hFont, Suicide);
	}
	
	static enum STOCK {
		ANSI_DIXED     = ANSI_FIXED_FONT,     /// Windows 文字セットの固定幅システムフォント 
		ANSI_VAR       = ANSI_VAR_FONT,       /// Windows 文字セットの可変幅システムフォント 
		DEVICE_DEFAULT = DEVICE_DEFAULT_FONT, /// デバイス依存のフォント（WindowsNT系OSのみ） 
		DEFAULT_GUI    = DEFAULT_GUI_FONT,    /// ユーザーインターフェイス用のデフォルトフォント（Windows9x系OSのみ） 
		OEM_FIXED      = OEM_FIXED_FONT,      /// OEM 文字セットの固定幅フォント 
		SYSTEM         = SYSTEM_FONT,         /// システムフォント 
	}
	this(STOCK Stock) {
		this(GetStockObject(Stock), false);
	}
	this(in LOGFONT* LogFont) {
		super(CreateFontIndirect(cast(LOGFONT*)LogFont), true);
	}

	/// フォントの高さ
	bool height(in int Height) {
		auto sFont=GetStruct(hFont);
		sFont.lfHeight = Height;
		return Import(sFont);
	}
	/// ditto
	const int height() {
		return GetStruct(hFont).lfHeight;
	}
	/// フォントの横幅
	bool width(in int Width) {
		auto sFont=GetStruct(hFont);
		sFont.lfWidth = Width;
		return Import(sFont);
	}
	/// ditto
	const int width() {
		return GetStruct(hFont).lfWidth;
	}
	/// 行の角度
	bool escapement(in int Escapement) {
		auto sFont=GetStruct(hFont);
		sFont.lfEscapement = Escapement;
		return Import(sFont);
	}
	/// ditto
	const int escapement() {
		return GetStruct(hFont).lfEscapement;
	}
	/// ベースラインの角度
	bool orientation(in int Orientation) {
		auto sFont=GetStruct(hFont);
		sFont.lfOrientation = Orientation;
		return Import(sFont);
	}
	/// ditto
	const int orientation() {
		return GetStruct(hFont).lfOrientation;
	}
	/// フォントの太さ
	static enum WEIGHT {
		DONTCARE   = FW_DONTCARE,   /// 
		THIN       = FW_THIN,       /// 
		EXTRALIGHT = FW_EXTRALIGHT, /// 
		ULTRALIGHT = FW_ULTRALIGHT, /// 
		LIGHT      = FW_LIGHT,      /// 
		NORMAL     = FW_NORMAL,     /// 普通。
		REGULAR    = FW_REGULAR,    /// 
		MEDIUM     = FW_MEDIUM,     /// 普通。
		SEMIBOLD   = FW_SEMIBOLD,   /// 
		DEMIBOLD   = FW_DEMIBOLD,   /// 
		BOLD       = FW_BOLD,       /// 太字。
		EXTRABOLD  = FW_EXTRABOLD,  /// 
		ULTRABOLD  = FW_ULTRABOLD,  /// 
		HEAVY      = FW_HEAVY,      /// 
		BLACK      = FW_BLACK,      /// 
	}
	/// フォントの太さ
	bool weight(WEIGHT Weight) {
		auto sFont=GetStruct(hFont);
		sFont.lfWeight = Weight;
		return Import(sFont);
	}
	/// ditto
	const WEIGHT weight() {
		return cast(WEIGHT)GetStruct(hFont).lfWeight;
	}
	/// イタリックフォント
	bool italic(bool Italic) {
		auto sFont=GetStruct(hFont);
		sFont.lfItalic = Italic;
		return Import(sFont);
	}
	/// ditto
	const bool italic() {
		return GetStruct(hFont).lfItalic == 1
			? true
			: false
		;
	}
	/// アンダーライン
	bool underLine(bool Underline) {
		auto sFont=GetStruct(hFont);
		sFont.lfUnderline = Underline;
		return Import(sFont);
	}
	/// ditto
	const bool underLine() {
		return GetStruct(hFont).lfUnderline == 1 ? true: false;
	}
	/// 取り消し線
	bool strikeOut(bool StrikeOut) {
		auto sFont=GetStruct(hFont);
		sFont.lfStrikeOut = StrikeOut;
		return Import(sFont);
	}
	/// ditto
	const bool strikeOut() {
		return GetStruct(hFont).lfStrikeOut == 1 ? true: false;
	}
	private void SetFaceName(in wchar[] Target, in Text text) {
		enforce(Target.length >= text.length-1, new FontException(Text("SetFaceName: %s", text.text)));
		lstrcpy(cast(wchar*)Target.ptr, text.ptr);
	}
	/// フォント名
	bool faceName(in Text FaceName) {
		auto sFont=GetStruct(hFont);
		SetFaceName(sFont.lfFaceName, FaceName);
		return Import(sFont);
	}
	/// ditto
	const Text faceName() {
		return Text(GetStruct(hFont).lfFaceName.ptr);
	}
	/// キャラクタセット
	bool charSet(ubyte CharSet) {
		auto sFont=GetStruct(hFont);
		sFont.lfCharSet = CharSet;
		return Import(sFont);
	}
	/// ditto
	const ubyte charSet() {
		return GetStruct(hFont).lfCharSet;
	}
	/// 出力精度
	static enum OUT: ubyte {
		CHARACTER_PRECIS = OUT_CHARACTER_PRECIS, /// 使用されません。 
		DEFAULT_PRECIS   = OUT_DEFAULT_PRECIS  , /// デフォルトの動作に任せます。 
		DEVICE_PRECIS    = OUT_DEVICE_PRECIS   , /// 同じ名前のフォントが見つかったときは、デバイス フォントを使用します。 
		OUTLINE_PRECIS   = OUT_OUTLINE_PRECIS  , /// 同じ名前のフォントが見つかったときは、TrueType や他のアウトラインベースのフォントを使用します（WindowsNT系OSのみ）。 
		RASTER_PRECIS    = OUT_RASTER_PRECIS   , /// 同じ名前のフォントが見つかったときは、ラスタ フォントを使用します。 
		STRING_PRECIS    = OUT_STRING_PRECIS   , /// 使用されません。 
		STROKE_PRECIS    = OUT_STROKE_PRECIS   , /// 同じ名前のフォントが見つかったときは、ベクトル フォントを使用します（Windows9x系OSのみ）。 
		TT_PRECIS        = OUT_TT_PRECIS       , /// 同じ名前のフォントが見つかったときは、TrueType フォントを使用します。 
		TT_ONLY_PRECIS   = OUT_TT_ONLY_PRECIS  , /// TrueType フォントを使用します。TrueType フォントが組み込まれていないときは、デフォルトの動作になります。 
	}
	/// 出力精度
	bool outPrecision(OUT OutPrecision) {
		auto sFont=GetStruct(hFont);
		sFont.lfOutPrecision = OutPrecision;
		return Import(sFont);
	}
	/// ditto
	const OUT outPrecision() {
		return cast(OUT)GetStruct(hFont).lfOutPrecision;
	}
	
	/// クリッピング精度
	static enum CLIP: ubyte {
		DEFAULT_PRECIS   = CLIP_DEFAULT_PRECIS   , /// デフォルトの動作に任せます。 
		CHARACTER_PRECIS = CLIP_CHARACTER_PRECIS , /// 使用されません。 
		STROKE_PRECIS    = CLIP_STROKE_PRECIS    , /// 使用されません。 
		MASK             = CLIP_MASK             , /// 使用されません。 
		EMBEDDED         = CLIP_EMBEDDED         , /// 読み込み専用の埋め込みフォントです。 
		LH_ANGLES        = CLIP_LH_ANGLES        , /// 座標系が右手座標系か左手座標系かによって、すべてのフォントの回転方向を決めるようにします。 
		TT_ALWAYS        = CLIP_TT_ALWAYS        , /// 使用されません。 
	}
	/// クリッピング精度
	bool clipPrecision(CLIP ClipPrecision) {
		auto sFont=GetStruct(hFont);
		sFont.lfClipPrecision = ClipPrecision;
		return Import(sFont);
	}
	/// ditto
	const CLIP clipPrecision() {
		return cast(CLIP)GetStruct(hFont).lfClipPrecision;
	}

	/// 出力品質
	static enum QUALITY: ubyte {
		DEFAULT = DEFAULT_QUALITY , /// フォントの文字品質は、重視されません。 
		DRAFT   = DRAFT_QUALITY   , /// フォントの文字品質は、PROOF_QUALITY フラグを指定したときほど重視されません。 
		PROOF   =PROOF_QUALITY    , /// フォントの文字品質は、論理フォントの属性を正確に一致させることよりも重視されます。 
	}
	/// 出力品質
	QUALITY quality(QUALITY Quality) {
		auto sFont=GetStruct(hFont);
		sFont.lfQuality = Quality;
		return cast(QUALITY)Import(sFont);
	}
	/// ditto
	const QUALITY quality() {
		return cast(QUALITY)GetStruct(hFont).lfQuality;
	}


	/// フォントのピッチ、およびファミリ
	static enum PITCH_FAMILY: ubyte {
		DEFAULT    = DEFAULT_PITCH  , /// ピッチはデフォルトの動作に任せます。 
		FIXED      = FIXED_PITCH    , /// ピッチは固定長になります。 
		VARIABLE   = VARIABLE_PITCH , /// ピッチは可変長になります。 
		DECORATIVE = FF_DECORATIVE  , /// 装飾付きフォントが指定されます（Old English など）。 
		DONTCARE   = FF_DONTCARE    , /// 一般的なファミリが指定されます。 
		MODERN     = FF_MODERN      , /// モノスペース フォントが指定されます（Pica、Elite、CourierNewなど）。 
		ROMAN      = FF_ROMAN       , /// セリフ（H、I などの上下にあるひげ飾り）のあるプロポーショナルフォントが指定されます（Times New Roman など）。 
		SCRIPT     = FF_SCRIPT      , /// 手書き風のデザインのフォントが指定されます（Script、Cursive など）。 
		SWISS      = FF_SWISS       , /// セリフ（H、I などの上下にあるひげ飾り）のないプロポーショナルフォントが指定されます。 
	}

	/// フォントのピッチ、およびファミリ
	bool pitchAndFamily(PITCH_FAMILY PitchAndFamily) {
		auto sFont=GetStruct(hFont);
		sFont.lfPitchAndFamily = PitchAndFamily;
		return Import(sFont);
	}
	/// ditto
	const PITCH_FAMILY pitchAndFamily() {
		return cast(PITCH_FAMILY)GetStruct(hFont).lfPitchAndFamily;
	}
}

