﻿/**

*/
module nemuxi.negui.draw.canvas;

debug import std.stdio: wl = writefln, pl = printf;

debug(canvas) void main() {}

import std.contracts;

import win32.windows;

import nemuxi.base;
import nemuxi.system.raii;
public import nemuxi.negui.draw.draw;
public import nemuxi.negui.draw.pen;
public import nemuxi.negui.draw.brush;
public import nemuxi.negui.draw.font;
public import nemuxi.negui.draw.color;
public import nemuxi.negui.draw.icon;
public import nemuxi.negui.draw.bitmap;
import nemuxi.negui.negui;

/**
*/
class Canvas: Draw {
	protected alias Handle hDC;
	protected NeGui gui;

	static enum TYPE {
		NEGUI,
		VIRTUAL,
		COPY,
	}
	private {
		TYPE Type;
	}
	

	/**
	*/
	override void Kill() {
		if(!hDC)
			throw new NemuxiException("デバイスコンテキストが未設定");

		if(oldPen) delPen();
		if(oldBrush) delBrush();
		if(oldFont) delFont();
		if(oldBitmap) delBitmap();

		switch(Type) {
			case TYPE.NEGUI:
				if(gui is null || !gui.alive()) {
					throw new NemuxiException("ウィンドウが死んでる");
				}
				if(!gui.releaseDC(hDC)) {
					break;
				}
				return;
			case TYPE.VIRTUAL:
				debug {
					auto r= DeleteObject(hDC) == FALSE;
					wl("Canvas.Kill.TYPE.VIRTUAL == %s", !r);
					if(r)
						break;
				} else {
					if(DeleteObject(hDC) == FALSE)
						break;
				}
				return;
			case TYPE.COPY:
				return;
			default:
				assert(false);
		}
		throw new NemuxiException("デバイスコンテキスト解放失敗");
	}

	/**
	*/
	this(HDC hDC, bool Suicide) {
		Type = TYPE.COPY;
		super(hDC, Suicide);
	}

	/**
	*/
	this(NeGui gui) {
		super(gui.getDC(), true);
		this.gui = gui;
		
		Type = TYPE.NEGUI;
	}
	/**
	*/
	this(SIZE* size) {
		auto TempDC=GetDC(GetDesktopWindow());
		scope(exit) ReleaseDC(GetDesktopWindow(), TempDC);
		
		super(CreateCompatibleDC(TempDC), true);
		
		auto hBitmap=CreateCompatibleBitmap(TempDC, size.cx, size.cy);
		scope(exit) DeleteObject(hBitmap);
		
		SelectObject(hDC, hBitmap);
		Type = TYPE.VIRTUAL;
	}
	/**
	*/
	TYPE type() {
		return Type;
	}
	
	protected {
		Pen pen;
		HPEN oldPen;
	}
	/**
	*/
	void setPen(Pen pen) {
		if(this.pen) {
			if(!delPen())
			throw new NemuxiException("ペン解放に失敗");
		}
		
		if(cast(bool)(this.pen = pen)) {
			oldPen = SelectObject(hDC, this.pen());
		}
	}
	/**
	*/
	bool delPen() {
		if(oldPen) {
			auto hPen=SelectObject(hDC, oldPen);
			
			oldPen = null;
			
			return pen() == hPen;
		}
		throw new NemuxiException("ペンを設定していない");
	}
	protected {
		Brush brush;
		HBRUSH oldBrush;
	}
	/**
	*/
	void setBrush(Brush brush) {
		if(this.brush) {
			if(!delBrush())
			throw new NemuxiException("ブラシ解放に失敗");
		}
		
		if(cast(bool)(this.brush = brush)) {
			oldBrush = SelectObject(hDC, this.brush());
		}
	}
	/**
	*/
	bool delBrush() {
		if(oldBrush) {
			auto hBrush=SelectObject(hDC, oldBrush);
			
			oldBrush = null;
			
			return brush() == hBrush;
		}
		throw new NemuxiException("ブラシを設定していない");
	}
	protected {
		Font font;
		HFONT oldFont;
	}
	/**
	*/
	void setFont(Font font) {
		if(this.font) {
			if(!delFont())
			throw new NemuxiException("フォント解放に失敗");
		}
		
		if(cast(bool)(this.font = font)) {
			oldFont = SelectObject(hDC, this.font());
		}
	}
	/**
	History:
		1.00β17:
			[S] 文言を修正。
	*/
	bool delFont() {
		if(oldFont) {
			auto hFont=SelectObject(hDC, oldFont);
			
			oldFont = null;
			
			return font() == hFont;
		}
		throw new NemuxiException("フォントを設定していない");
	}

	protected {
		Bitmap bitmap;
		HBITMAP oldBitmap;
	}
	/**
	*/
	void setBitmap(Bitmap bitmap) {
		if(this.bitmap) {
			if(!delBitmap())
			throw new NemuxiException("ビットマップ解放に失敗");
		}
		
		if(cast(bool)(this.bitmap = bitmap)) {
			oldBitmap = SelectObject(hDC, this.bitmap());
		}
	}
	/**
	*/
	bool delBitmap() {
		if(oldBitmap) {
			auto hBitmap=SelectObject(hDC, oldBitmap);
			
			oldBitmap = null;
			
			return bitmap() == hBitmap;
		}
		throw new NemuxiException("ビットマップを設定していない");
	}

	/**
	*/
	SIZE size() {
		auto Rect=rect();
		SIZE Size=void;
		Size.cx = Rect.right - Rect.left;
		Size.cy = Rect.bottom - Rect.top;
		return Size;
	}
	/**
	*/
	RECT rect() {
		RECT Rect=void;
		GetClipBox(hDC, &Rect);
		return Rect;
	}
	/**
	*/
	void getRect(RECT* Rect) {
		GetClipBox(hDC, Rect);
	}

	COLOR textColor() {
		auto color=GetTextColor(hDC);
		if(color == CLR_INVALID)
			throw new NemuxiException("文字色取得失敗");
		return COLOR(color);
	}
	COLOR textColor(COLOR color) {
		return COLOR(SetTextColor(hDC, color.code));
	}

	/**
	*/
	COLOR backColor() {
		auto color=GetBkColor(hDC);
		if(color == CLR_INVALID)
			throw new NemuxiException("背景色取得失敗");
		return COLOR(color);
	}
	/**
	*/
	COLOR backColor(COLOR color) {
		return COLOR(SetBkColor(hDC, color.code));
	}
	/**
	*/
	static enum BACKMODE {
		OP = OPAQUE , ///選択されている背景色を利用し、背景を塗りつぶします。
		TR = TRANSPARENT, ///背景をそのまま残します。
	}
	/**
	*/
	BACKMODE backMode() {
		auto mode=GetBkMode(hDC);

		if(mode == 0)
			throw new NemuxiException("背景モード取得失敗");

		return cast(BACKMODE)mode;
	}
	/**
	*/
	BACKMODE backMode(BACKMODE Mode) {
		auto mode=SetBkMode(hDC, Mode);
		
		if(mode == 0)
			throw new NemuxiException("背景モード取得失敗");
		
		return cast(BACKMODE)mode;
	}

	bool getTextArea(in Text text, out SIZE size) {
		return GetTextExtentPoint32(hDC, text.ptr, text.length, &size) ?
			true: false;
	}
	/**
	History:
		1.00β17:
			新規追加。
	*/
	ref SIZE getTextArea(in Text text) {
		auto Size=new SIZE;
		if(GetTextExtentPoint32(hDC, text.ptr, text.length, cast(SIZE*)Size)) {
			return *Size;
		}
		throw new NemuxiException(Err.toString);
	}

	int letterSpacing() {
		auto len=GetTextCharacterExtra(hDC);
		if(len == 0x8000000)
			throw new NemuxiException("文字間隔取得失敗");
		return len;
	}
	int letterSpacing(int Length) {
		auto len=SetTextCharacterExtra(hDC, Length);
		if(len == 0x8000000)
			throw new NemuxiException("前回文字間隔取得失敗");
		return len;
	}

	/**
	*/
	bool drawText(in Text text, int x, int y) {
		return TextOut(hDC, x, y, text.ptr, text.length) == TRUE ?
			true: false;
	}
	/**
	*/
	static enum FORMAT {
		BOTTOM          = DT_BOTTOM, /// 長方形の下辺にテキストを揃えます。DT_SINGLELINEと同時に指定する必要があります。 
		CALCRECT        = DT_CALCRECT, /// テキストを表示するために必要な長方形の大きさを lpRect パラメータに格納します。テキストは描画されません。 
		CENTER          = DT_CENTER, /// テキストを水平方向に中央揃えで表示します。 
		EDITCONTROL     = DT_EDITCONTROL, /// 複数行エディット コントロールがもつ特性と同じ特性で描画を行います。 
		END_ELLIPSIS    = DT_END_ELLIPSIS, /// 指定した長方形領域にテキストが収まらない場合、テキストの最後を省略符号 (...) に置き換えます。 
		PATH_ELLIPSIS   = DT_PATH_ELLIPSIS, /// 指定した長方形領域にテキストが収まらない場合、テキストの途中を省略符号 (...) に置き換えます。 
		EXPANDTABS      = DT_EXPANDTABS, /// タブ文字を展開します。 
		EXTERNALLEADING = DT_EXTERNALLEADING, /// 行の高さに、外部レディングの高さ（行間として適当な高さ）を加えます。 
		LEFT            = DT_LEFT, /// テキストを左揃えにします。 
		MODIFYSTRING    = DT_MODIFYSTRING, /// DT_END_ELLIPSIS フラグ、DT_PATH_ELLIPSIS フラグによって変更された文字列をlpStringに格納します。 
		NOCLIP          = DT_NOCLIP, /// クリッピングを行わないようにします。 
		NOPREFIX        = DT_NOPREFIX, /// プリフィックス文字（「&」を次の文字のアンダースコアに置き換える、「&&」を「&」として処理する）の処理を行わないようにします。 
		RIGHT           = DT_RIGHT, /// テキストを右揃えにします。 
		RTLREADING      = DT_RTLREADING, /// 右から左に向かってテキストを表示します。ヘブライ語、アラビア語などを表示する場合に利用します。 
		SINGLELINE      = DT_SINGLELINE, /// テキストを改行せず、一行で表示します。 
		TABSTOP         = DT_TABSTOP, /// タブ文字の空白文字数を設定します。このフラグを指定した場合は uFormat パラメータの15ビットから8ビット（下位ワードの上位バイト）で空白文字数を設定します。 
		TOP             = DT_TOP, /// 長方形の上辺にテキストを揃えます。DT_SINGLELINEと同時に指定する必要があります。 
		VCENTER         = DT_VCENTER, /// テキストを垂直方向の中央揃えで表示します。DT_SINGLELINEと同時に指定する必要があります。 
		WORDBREAK       = DT_WORDBREAK, /// テキストを複数行で表示します。折り返しは自動的に行われます。 
	}
	/**
	History:
		1.00β11:
			引数の型変更(RECT* -> ref RECT)
	*/
	int drawText(in Text text, ref RECT Rect, FORMAT Format) {
		return DrawText(hDC, text.ptr, text.length, &Rect, Format);
	}
	
	/**
	*/
	POINT position() {
		POINT point=void;
		
		if(GetCurrentPositionEx(hDC, &point) == TRUE)
			return point;
		throw new NemuxiException("現在位置情報取得失敗");
	}
	//　前の位置
	POINT position(POINT* Pos) {
		POINT point=void;
		
		if(MoveToEx(hDC, Pos.x, Pos.y, &point) == TRUE)
			return point;
		throw new NemuxiException("前回位置設定取得失敗");
	}

	POINT positionBrush() {
		POINT point=void;
		if(GetBrushOrgEx(hDC, &point) == FALSE)
			throw new NemuxiException("ブラシ原点取得失敗");
		return point;
	}
	POINT positionBrush(POINT* point) {
		POINT oldPoint=void;
		if(SetBrushOrgEx(hDC, point.x, point.y, cast(POINT*)&oldPoint) == FALSE)
			throw new NemuxiException("ブラシ原点設定失敗");
		return oldPoint;
	}

	

	/**
	直線描画
	*/
	bool drawLine(POINT* Pos) {
		return cast(bool)LineTo(hDC, Pos.x, Pos.y);
	}
	/**
	直線描画
	
	History:
		1.00β19:
			[P] Logger削除。
	*/
	bool drawLine(POINT* Start, POINT* End) {
		try {
			auto Pos=position(Start);
			scope(success) position(&Pos);
			return drawLine(End);
		} catch(NemuxiException e) {
			return false;
		}
	}
	/// ditto
	bool drawLine(POINT[] polygons) {
		return cast(bool)Polyline(hDC, cast(POINT*)polygons.ptr, polygons.length);
	}
	/**
	矩形描画
	*/
	bool drawRectangle(RECT* Rect) {
		return cast(bool)Rectangle(hDC, Rect.left, Rect.top, Rect.right, Rect.bottom);
	}
	/**
	矩形塗潰し。

	Params:
		Rect = 塗潰し範囲。

		brush = 塗潰しに使用するブラシ。
		        nullであれば現在のブラシを使用。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。
	*/
	bool fillRect(ref const(RECT) Rect, Brush brush=null) {
		return cast(bool)FillRect(hDC, cast(RECT*)&Rect, brush ? brush(): null);
	}
	bool rectangle(int Left, int Top, int Right, int Bottom) {
		return cast(bool)Rectangle(hDC, Left, Top, Right, Bottom);
	}
	bool rectangle(RECT* Rect) {
		return rectangle(Rect.left, Rect.top, Rect.right, Rect.bottom);
	}

	/**
	*/
	static enum PAT {
		COPY  = PATCOPY  , /// 指定されたパターンを描画先ビットマップへコピーします。
		PAT   = PATINVERT, /// 論理 XOR 演算子を使って、指定されたパターンの色と描画先の色を組み合わせます。
		INV   = DSTINVERT, /// 描画先長方形の色を反転します。
		BLACK = BLACKNESS, /// 物理パレットのインデックス 0 に対応する色（既定の物理パレットでは黒）で、描画先の長方形を塗りつぶします。
		WHITE = WHITENESS, /// 物理パレットのインデックス 1 に対応する色（既定の物理パレットでは白）で、描画先の長方形を塗りつぶします。
	}
	/**
	*/
	bool patRect(RECT* Rect, PAT pat) {
		return cast(bool)PatBlt(
			hDC,
			Rect.left, Rect.top,
			Rect.right, Rect.bottom,
			pat
		);
	}
	/**
	*/
	bool patRect(PAT pat) {
		auto box=rect();
		return patRect(&box, pat);
	}

	bool invert(RECT* Rect) {
		return cast(bool)InvertRect(hDC, Rect);
	}

	/**
	矩形フォーカス描画。
	
	History:
		1.00β17:
			新規作成。
	*/
	bool drawFocusRect(const ref RECT Rect) {
		return cast(bool)DrawFocusRect(hDC, cast(RECT*)&Rect);
	}

	/**
	*/
	bool drawIcon(Icon icon, int x, int y) {
		return cast(bool)DrawIcon(hDC, x, y, icon());
	}
	/**
	*/
	bool drawIcon(Icon icon, POINT* Point, int size, Brush brush=null) {
		return cast(bool)DrawIconEx(
			hDC,
			Point.x, Point.y,
			icon(),
			size, size,
			0,
			brush ? brush(): null,
			DI_NORMAL
		);
	}
	/**
	*/
	bool drawState(Icon icon, POINT* Point, int size, bool Enable) {
		return cast(bool)DrawState(
			hDC,
			null,
			&DrawIcon_Proc,
			cast(LPARAM)icon(),
			0,
			Point.x,
			Point.y,
			size,
			size,
			DST_COMPLEX | (Enable ? DSS_NORMAL: DSS_DISABLED)
		);
	}
	/**
	*/
	bool drawState(in Text text, POINT* Point, bool Enable, bool Prefix) {
		return cast(bool)DrawState(
			hDC,
			null,
			null,
			cast(LPARAM)text.ptr,
			text.length,
			Point.x,
			Point.y,
			0,
			0,
			(Prefix ? DST_PREFIXTEXT: DST_TEXT) | (Enable ? DSS_NORMAL: DSS_DISABLED)
		);
	}

	static enum ROC {
		BLACKNESS      , /// 物理パレットのインデックス 0 に対応する色（既定の物理パレットでは黒）で、コピー先の長方形を塗りつぶします。
		CAPTUREBLT     , /// Windows 98 と Windows 2000：アプリケーションのウィンドウより上位にレイヤー化されているすべてのウィンドウを、最終的なイメージに含めます。既定では、アプリケーションのウィンドウだけがイメージに含まれます。
		DSTINVERT      , /// コピー先長方形の色を反転します。
		MERGECOPY      , /// 論理 AND 演算子を使って、コピー元の色とコピー先の色を組み合わせます。
		MERGEPAINT     , /// 論理 OR 演算子を使って、コピー元の色を反転した色と、コピー先の色を組み合わせます。
		NOMIRRORBITMAP , /// Windows 98 と Windows 2000：ビットマップのミラーリング（ミラーイメージを作成すること）を防止します。
		NOTSRCCOPY     , /// コピー元の色を反転して、コピー先へコピーします。
		NOTSRCERASE    , /// 論理 OR 演算子を使って、コピー元の色とコピー先の色を組み合わせ、さらに反転します。
		PATCOPY        , /// 指定したパターンをコピー先へコピーします。
		PATINVERT      , /// 論理 XOR 演算子を使って、指定したパターンの色と、コピー先の色を組み合わせます。
		PATPAINT       , /// 論理 OR 演算子を使って、指定したパターンの色と、コピー元の色を反転した色を組み合わせます。さらに論理 OR 演算子を使って、その結果と、コピー先の色を組み合わせます。
		SRCAND         , /// 論理 AND 演算子を使って、コピー元の色とコピー先の色を組み合わせます。
		SRCCOPY        , /// コピー元長方形をコピー先長方形へそのままコピーします。
		SRCERASE       , /// 論理 AND 演算子を使って、コピー先の色を反転した色と、コピー元の色を組み合わせます。
		SRCINVERT      , /// 論理 XOR 演算子を使って、コピー元の色とコピー先の色を組み合わせます。
		SRCPAINT       , /// 論理 OR 演算子を使って、コピー元の色とコピー先の色を組み合わせます。
		WHITENESS      , /// 物理パレットのインデックス 1 に対応する色（既定の物理パレットでは白）で、コピー先の長方形を塗りつぶします。
	}
	/**
	History:
		1.00β19:
			[S] 引数の属性を一部変更。
	*/
	bool paste(const ref POINT TargetPoint, const ref SIZE TargetSize, in Canvas Src, const ref POINT SrcPoint, ROC Roc) {
		return cast(bool)BitBlt(
			hDC,
			TargetPoint.x,
			TargetPoint.y,
			TargetSize.cx,
			TargetSize.cy,
			Src.hDC,
			SrcPoint.x,
			SrcPoint.y,
			Roc
		);
	}

}
extern(Windows) private BOOL DrawIcon_Proc(HDC hDC, LPARAM lParam, WPARAM wParam, int cx, int cy) {
	return cast(bool)DrawIconEx(hDC, 0, 0, cast(HICON)lParam, cx, cy, 0, null, DI_NORMAL);
}




