﻿/**
アイコン拡張。

nemuxi.negui.draw.iconを使用してちょっとやれること増やしたよん。

Bugs:
	一部NeGuiに統合すべき。
*/
module nemuxi.image.icon;

debug import std.stdio: wl = writefln, pl = printf;

import std.contracts;

import win32.windows;

import nemuxi.base;
import nemuxi.file.file;

/**
アイコンサイズ。
*/
enum ICONSIZE {
	SMALL , /// 16*16アイコン。
	NORMAL, /// 32*32アイコン。
}
public import nemuxi.negui.draw.icon;
/**
めんどいから縦横一緒。
てか一緒じゃない状態を想像したくない。

History:
	1.00β19:
		[S] final switchの使用。
*/
pure nothrow int GetIconSize(ICONSIZE IconSize) {
	final switch(IconSize) {
		case ICONSIZE.SMALL:  return 16;
		case ICONSIZE.NORMAL: return 32;
	}
}
debug(icon) unittest {
	assert(GetIconSize(ICONSIZE.SMALL)  == 16);
	assert(GetIconSize(ICONSIZE.NORMAL) == 32);
}

/**
アイコンを取得。

ABのGetIconHandleそのまま。

Params:
	Address = ファイルアドレス。

	IconIndex = アイコンインデックス。
	            Addressがアイコンを持っている(実行形式)の場合に
	            IconIndexがAddressより大きい場合は0に設定される。

	IconSize = アイコンの大きさ。

Return:
	アイコン。

Exception:
	ミスった場合はNemuxiException、たんまにWin32 Exception。
	Addressが行方不明の場合もなげる。

In:
	Address.length > 0

History:
	1.00β11:
		DLLのアイコンを取得できていなかった不具合の修正。
		ファイルの有無で例外を投げる。
*/
Icon GetFileIcon(in Text Address, int IconIndex, ICONSIZE IconSize)
in {
	assert(Address.length);
}
body {
	enforce(FILE.isExistence(Address), new NemuxiException(Address ~ "がない"));
	SHFILEINFO FileInfo=void;

	const pAddress=Address.ptr;
	immutable dll=PATH.cmp(PATH.getExtension(Address), Text("dll"))==0;

	//ファイルは実行形式？
	bool IsBin=false;
	int IconCount;
	if(dll || SHGetFileInfo(pAddress, 0, &FileInfo, FileInfo.sizeof, SHGFI_EXETYPE)) {
		IsBin = true;
		//保持アイコン総数
		IconCount=ExtractIconEx(pAddress, -1, null, null, 0);
		if(!IconCount && dll) {
			// DLLだけどアイコンを保持していない。
			IsBin = false;
		}
	}
	if(IsBin) {
		// ファイルは実行形式
		if(IconIndex > IconCount) {
			// 指定アイコンインデックスが多すぎるので最小値へ
			IconIndex = 0;
		}

		with(ICONSIZE) switch(IconSize) {
			case SMALL:
				ExtractIconEx(pAddress, IconIndex, null, &FileInfo.hIcon, 1);
				break;
			case NORMAL:
				ExtractIconEx(pAddress, IconIndex, &FileInfo.hIcon, null, 1);
				break;
			default:
				assert(false);
		}
	} else {
		// ふつーのファイル
		int IconFlag=void;
		with(ICONSIZE) switch(IconSize) {
			case SMALL:
				IconFlag = SHGFI_SMALLICON;
				break;
			case NORMAL:
				IconFlag = SHGFI_LARGEICON;
				break;
			default:
				assert(false);
		}
		if(!SHGetFileInfo(pAddress, FILE_ATTRIBUTE_ARCHIVE, &FileInfo, FileInfo.sizeof, SHGFI_ICON | IconFlag)) {
			throw new NemuxiException("GetFileIcon:SHGetFileInfo(" ~ Address.toString() ~ ")");
		}
	}
	
	return new Icon(FileInfo.hIcon, true);
}

debug(icon) unittest {
	GetFileIcon(Text(`C:\Program Files\Mozilla Firefox\plugins`),0,ICONSIZE.SMALL);
}
/**

Bugs:
	アイコン名と内容が多分違う。
*/
enum SYSICON: ushort {
	APPLICATION = OIC_SAMPLE     ,
	ASTERISK    = OIC_INFORMATION       ,
	EXCLAMATION = OIC_WARNING       ,
	HAND        = OIC_HAND    ,
	QUESTION    = OIC_QUES    ,
	WINLOGO     = OIC_WINLOGO,

	NEMUXI = ushort.max-100,
	URI,
	MULTI,
	TASKTRAY,
	
	FOLDER,
	FILE,
}
private invariant LR_SHARD = 0x8000;
/**
システムアイコンの取得。

当moduleの「システムアイコン」は

①Windows内部アイコン<br />
②ﾈﾑぃのリソースアイコン<br />
③②が無効な場合にﾈﾑぃの内部リソースアイコン

を指す。

Params:
	SysIcon = アイコンフラグ。

	IconSize = アイコンサイズ。

Return:
	アイコン。

Exception:
	Windows内部アイコンの取得に失敗した場合にNemuxiExceptionを投げる。
*/
Icon GetSystemIcon(SYSICON SysIcon, ICONSIZE IconSize) {
	auto size=GetIconSize(IconSize);
	if(SysIcon < SYSICON.NEMUXI) { // システムアイコン。
		HICON hIcon=LoadImage(
			null,
			MAKEINTRESOURCE(SysIcon),
			IMAGE_ICON,
			size, size,
			LR_SHARD
		);
		if(!hIcon) {
			throw new NemuxiException("GetSystemIcon:LoadIcon");
		}
		return new Icon(hIcon, false);
	}
	Text IconName=void;
	switch(SysIcon) {
		case SYSICON.NEMUXI:   IconName = "nemuxi.ico";   break;
		case SYSICON.URI:      IconName = "uri.ico";      break;
		case SYSICON.MULTI:    IconName = "multi.ico";    break;
		case SYSICON.TASKTRAY: IconName = "tasktray.ico"; break;
		
		case SYSICON.FOLDER:   IconName = "folder.ico";   break;
		case SYSICON.FILE:     IconName = "file.ico";     break;
		
		default: assert(false);
	}
	Text IconAddress=PATH.addFolderSep(StaticData.resourceFolder) ~ IconName;
	if(FILE.isExistence(IconAddress)) {
		return GetFileIcon(IconAddress, 0, IconSize);
	} else {
		return new Icon(LoadImage(
			GetModuleHandle(null),
			GetIconResource(SysIcon),
			IMAGE_ICON,
			size, size,
			LR_SHARD
		), false);
	}
	
}
private wchar* GetIconResource(SYSICON SysIcon) {
	int IconName;
	switch(SysIcon) {
		case SYSICON.NEMUXI:   IconName = 100; break;
		case SYSICON.URI:      IconName = 106; break;
		case SYSICON.MULTI:    IconName = 108; break;
		case SYSICON.TASKTRAY: IconName = 105; break;
		
		case SYSICON.FOLDER:   IconName = 110; break;
		case SYSICON.FILE:     IconName = 112; break;
		
		default: assert(false);
	}
	return cast(wchar*)IconName;
}
/+
#define IDI_ICON1                               100
#define IDI_TASKTRAY                            105
#define IDI_URI                                 106
#define IDI_MULTI                               108
#define IDI_FOLDER                              110
#define IDI_FILE                                112
+/

