﻿/**
アイテム関数。
*/
module nemuxi.file.items.itemfunc;

debug import std.stdio: wl = writefln, pl = printf;

debug(itemfunc) void main() {}

import std.string;
import std.conv;
import std.file;
import std.path;

import win32.windef;

import etc.kareki.kareki;

import nemuxi.base;
import nemuxi.file.file;
import nemuxi.file.items.item;
import nemuxi.file.items.linkitem;
public import nemuxi.negui.system.timer;
import nemuxi.image.icon;
import nemuxi.utility.convert.env;

/**
*/
class ItemDateTime: DateTime {
	invariant() {
		assert(Mask == MASK.DATE+MASK.TIME);
		assert(0 <= SystemTime.wYear && SystemTime.wYear <= 9999);
		assert(Format    == FORMAT);
		assert(MonthList == MONTHLIST);
	}

	invariant FORMAT    = "[YYYY]/[M]/[DD] [hh]:[mm]:[ss]";
	invariant MONTHLIST = ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"];
	invariant OUTSTRING = "YYYY/MM/DD HH:MM:SS";
	
	/**
	*/
	this() {
		super();
		Format    = FORMAT;
		MonthList = MONTHLIST;
	}
	/**
	*/
	this(WORD Year, WORD Month, WORD Day, WORD Hour, WORD Minute, WORD Second, WORD MilliSecond) {
		this();
		
		year   = Year;
		month  = Month;
		day    = Day;
		hour   = Hour;
		minute = Minute;
		second = Second;
		ms     = MilliSecond;
		ChangeWeek();
	}
	/**
	History:
		1.00β19:
			[P] 日時情報のチェックを二重に行っていた部分を一本化。
	*/
	this(Text date, lazy Text ItemInfo) {
		this();
		

		try {
			if(date.length != OUTSTRING.length) {
				throw new ItemException(Text("内部保持用日時情報長が不正, [%s = %s](%s)", date.text8, date.length, ItemInfo.text8), EC.ITEM_DATE_LENGTH);
			}
			
			/+
			auto time=split(date);
			auto ymd=split(time[0], "/");
			auto hms=split(time[1], ":");
			+/
			auto time=date.split(' ');
			auto ymd=time[0].split('/');
			auto hms=time[1].split(':');
			
			if(ymd.length + hms.length != 6)
				throw new ItemException("日時情報分割数が不正。", EC.ITEM_DATE_SPLIT);
			
			year   = to!(WORD)(ymd[0].text);
			month  = to!(WORD)(ymd[1].text);
			day    = to!(WORD)(ymd[2].text);
			hour   = to!(WORD)(hms[0].text);
			minute = to!(WORD)(hms[1].text);
			second = to!(WORD)(hms[2].text);

			/+
			bool check=true;
			check &= year > 0 && year < 10000;
			check &= month >= 1 && month <= 12;
			check &= day >= 1 && day <= 31;
			check &= hour >= 0 && hour <= 23;
			check &= minute >= 0 && minute <= 59;
			check &= second >= 0 && second <= 59;
			if(!check)
				throw new ItemException("日時情報がへんてこりん", EC.ITEM_DATE_INFO);
			+/
			ChangeWeek();

		} catch(Exception e) {
			throw new ItemException("日時情報変換時に例外", EC.ITEM_DATE, e);
		}
	}
	
	override void year(in WORD Year) {
		if(0 <= Year && Year <= 9999) {
			super.year(Year);
		} else {
			throw new ItemException("年情報が不正", EC.ITEM_DATE);
		}
	}
	override const WORD year()
	out(r) {
		assert(0 <= r && r <= 9999);
	}
	body {
		return super.year();
	}
}


debug(itemfunc) unittest {
	ItemDateTime date1=new ItemDateTime();
	ItemDateTime date2;
	with(date1) {
		year   = 2009;
		month  = 1;
		day    = 1;
		hour   = 0;
		minute = 0;
		second = 0;
	}
	assert(date1.toString == "2009/01/01 00:00:00", date1.toString);
	date2 = new ItemDateTime(Text("2009/01/01 00:00:00"));
	assert(date1.toString == date2.toString);
	with(date1) {
		year   = 1;
		month  = 12;
		day    = 31;
		hour   = 23;
		minute = 59;
		second = 59;
	}
	assert(date1.toString == "0001/12/31 23:59:59");
	date2 = new ItemDateTime(Text("0001/12/31 23:59:59"));
	assert(date1.toString == date2.toString);

	try {
		new ItemDateTime(Text("005/112/31 23:59:59"));
		assert(false);
	} catch(NemuxiException) {
		assert(true);
	}
}
/+
Text GetName(Item item)
in {
	assert(item);
}
body {
	return item.name;
}
+/

/**
親フォルダの取得。

Params:
	item = 親フォルダを取得するアイテム。
	       <table>
	         <thead>
	           <tr>
	             <th>Item.TYPE</th>
	             <th>フォルダ</th>
	           </tr>
	         </thead>
	         <tbody>
	           <tr>
	             <th>NORMAL</th>
	             <td>親フォルダ</td>
	           </tr>
	           <tr>
	             <th>URI</th>
	             <td>
	               iten.uriが有効なローカルアドレスならその親フォルダ。
	               無効な場合はスペース分割("が存在する場合は括弧として扱う)した第一要素をローカルアドレスとする。
	               分割要素が無効であれば作業フォルダを使用、それも無理ならnullを返す。
	             </td>
	           </tr>
	           <tr>
	             <th>MULTI</th>
	             <td>
	               作業フォルダを使用。
	               作業フォルダが有効でなければnull。
	             </td>
	           </tr>
	           <tr>
	             <th>LINK</th>
	             <td></td>
	           </tr>
	         </tbody>
	       </table>
*/
Text GetFolder(Item item, lazy Kareha ItemBase=null)
in {
	assert(item);
}
out(r) {
	if(r.length)
	assert(r[r.length-1]=='\\', r.toString);
}
body {
	
	with(Item.TYPE) switch(item.type()) {
		case NORMAL: {
			return PATH.getOwnerFolder(item.address);
		}
		case URI: {
			try {
				auto Folder = item.workFolder;
				
				if(FILE.isFolder(Folder)) {
					return PATH.addFolderSep(Folder);
				} else {
					return PATH.getOwnerFolder(Folder);
				}
			} catch(NemuxiFileException e) {
				if(FILE.isExistence(item.uri)) {
					return PATH.getOwnerFolder(item.uri);
				} else if(item.uri.length) {
					char   SplitChar;
					size_t StartIndex;
					if(item.uri[0] == '"') {
						SplitChar  = '"';
						StartIndex = 1;
					} else {
						SplitChar  = ' ';
						StartIndex = 0;
					}
					for(auto i=StartIndex; i < item.uri.length; i++) {
						if(item.uri[i] == SplitChar) {
							auto Folder=PATH.getOwnerFolder(item.uri[StartIndex..i]);
							if(FILE.isFolder(Folder)) {
								return PATH.addFolderSep(Folder);
							}
							break;
						}
					}
				}
			}
			
			return Text.emptyText;
		}
		case MULTI: {
			if(!item.workFolder.length) {
				return Text.emptyText;
			}
			auto Folder = PATH.addFolderSep(item.workFolder());
			
			try {
				return FILE.isFolder(Folder)
					? Folder
					: Text.emptyText
				;
			} catch(NemuxiFileException e) {
				Text.emptyText;
			}
		}
		case LINK: {
			auto link=new LinkItem(item, ItemBase);
			return GetFolder(link, ItemBase);
		}
		default:
			assert(false);
	}
}
debug(itemfunc) unittest {
	auto aki=newItem();
	auto item=new Item(Text("ie"), aki[`Items`]);
}

Text GetWorkFolder(Item item, lazy Kareha ItemBase=null)
in {
	assert(item);
}
out(r) {
	if(r.length)
	assert(r[r.length-1]=='\\', r.toString);
}
body {
	with(Item.TYPE) switch(item.type()) {
		case NORMAL:
			if(item.workFolder.length) {
				return PATH.addFolderSep(item.workFolder);
			} else {
				auto Folder=PATH.getOwnerFolder(item.address);
				if(Folder.length) {
					return PATH.addFolderSep(Folder);
				}
				
				return Text.emptyText;
			}
		case URI, MULTI:
			if(item.workFolder.length) {
				return PATH.addFolderSep(item.workFolder);
			} else {
				return Text.emptyText;
			}
		case LINK:
			return GetWorkFolder(new LinkItem(item, ItemBase), ItemBase);
		default:
			assert(false);
	}
}

/**
History:
	1.00β16:
		[B] 環境変数の処理が甘かった部分を修正。
	
	1.00β15:
		[P] 無効アイコン表示をSYSICON.ASTERISKからSYSICON.EXCLAMATIONに変更。まぁこの列挙体は何の考慮もしてないからあてにならないけど。
		[B] 環境変数を含んでいた場合に起こる不具合の修正。
*/
Icon GetIcon(Item item, ICONSIZE IconSize, lazy Kareha ItemBase=null)
in {
	assert(item);
}
body {
	Text IconAddress=void;
	if(item.type() != Item.TYPE.LINK)
	IconAddress=toPlain(item.iconAddress());
	
	switch(item.type()) {
		case Item.TYPE.LINK:
			return GetIcon(new LinkItem(item, ItemBase), IconSize, ItemBase);
		case Item.TYPE.NORMAL:
			if(!FILE.isExistence(IconAddress)) {
				Text FileAddress=toPlain(item.address());
				if(FILE.isExistence(FileAddress)) {
					IconAddress = FileAddress;
				} else {
					IconAddress = Text.emptyText();
				}
			}
			break;
		case Item.TYPE.URI, Item.TYPE.MULTI:
			if(!FILE.isExistence(IconAddress)) {
				IconAddress = Text.emptyText();
			}
			break;
		default:
			assert(false);
	}
	if(IconAddress.length) {
		return GetFileIcon(IconAddress, item.iconIndex(), IconSize);
	} else {
		SYSICON SysIcon;
		if(item.type() == Item.TYPE.NORMAL) {
			SysIcon = SYSICON.EXCLAMATION;
		} else {
			assert(item.type() != Item.TYPE.LINK);
			SysIcon = item.type() == Item.TYPE.URI ? SYSICON.URI:SYSICON.MULTI;
		}
		return GetSystemIcon(SysIcon, IconSize);
	}
}

