﻿/**
アイテム関連のグループ。

ここから直接アイテムに結びつくことはそんなに無いはず。
*/
module nemuxi.file.items.group;

debug import std.stdio: wl = writefln, pl = printf;

import std.contracts;

public import etc.kareki.kareki;

import nemuxi.base;
import nemuxi.system.application;
import nemuxi.file.data;
import nemuxi.file.items.item;
import nemuxi.file.items.linkitem;
import nemuxi.image.icon;

///
class GroupDataException: DataException {
	mixin MixInNemuxiException;
}

/**
グループ。

History:
	1.00β15:
		[S] Textをベースに作り直し。
		[S] クラス名変更。
*/
class GroupBase: Data {
	invariant() {
		assert(ks[GROUP].type == HAGATA.TSV_String);
	}
	enum {
		GROUP = 0
	}
	
	/**
	コンストラクタ。
	
	Params:
		GroupTree = グループ。
		            こいつがなくちゃ表明違反。
	*/
	this(Kareha GroupTree)
	in {
		assert(GroupTree);
	}
	body {
		ks = new Kareha[GROUP+1];

		// グループ設定。
		ks[GROUP] = GroupTree;
		if(!(ks[GROUP].have() && ks[GROUP].type == HAGATA.TSV_String)) {
			ks[GROUP] = (string[]).init;
		}
	}

	/**
	自身の要素数を取得。

	Return:
		要素数。
	*/
	const size_t baseLength() {
		return ks[GROUP].tsvLength;
	}

	/**
	*/
	const Text baseName(size_t Index) {
		try {
			return ks[GROUP].unSafe!(string[])[Index].toText;
		} catch(Error e) {
			throw new GroupDataException(Text("GROUP.length = %s", ks[GROUP].tsvLength), EC.NONE, e);
		}
	}
	/**
	*/
	const Text[] baseNames() {
		return Texts(ks[GROUP].unSafe!(string[]));
	}
	/**
	連想配列的なチェック。
	
	あくまでチェック。
	*/
	const bool inBaseName(in Text BaseName) {
		const names=baseNames;

		foreach(i, name; names) {
			if(name.text == BaseName.text) {
				return true;
			}
		}

		return false;
	}

	protected {
		/// 子アイテムの取得。
		const Kareha Child(size_t Index)
		in {
			assert(baseLength >= Index);
		}
		body {
			try {
				immutable ChildName=ks[GROUP].unSafe!(string[])[Index];
				return ks[GROUP][ChildName];
			} catch(Throwable e) {
				throw new GroupDataException(Text("Index = %s", Index), EC.NONE, e);
			}
		}
		const Kareha Child(Text GroupName)
		in {
			assert(GroupName.length);
		}
		body {
			try {
				return ks[GROUP][GroupName.toString];
			} catch(KarekiException e) {
				throw new GroupDataException(Text("GroupName = %s", GroupName.text), EC.NONE, e);
			}
		}
	}
	/// ditto
	size_t childLength(size_t Index) {
		auto Base=Child(Index);
		return Base.tsvLength;
	}
	size_t childLength(Text GroupName) {
		auto Base=Child(GroupName);
		return Base.tsvLength;
	}
	
	/**
	*/
	const Text[] childNames(size_t Index) {
		try {
			auto Base=Child(Index);
			return Texts(Base.unSafe!(string[]));
		} catch(Throwable e) {
			throw new GroupDataException(Text("childNames.length = %s", ks[GROUP].tsvLength), EC.NONE, e);
		}
	}
	/**
	*/
	const Text[] childNames(Text GroupName) {
		try {
			auto Base=Child(GroupName);
			return Texts(Base.unSafe!(string[]));
		} catch(Throwable e) {
			throw new GroupDataException(Text("%s in GroupName", GroupName.toString), EC.NONE, e);
		}
	}
}

class Main: GroupBase {
	///
	this(Kareha MainGroup) {
		super(MainGroup);
	}

	Sub subGroup(Text SubGroupName) {
		try {
			return new Sub(this, ks[0][SubGroupName.toString]);
		} catch(Exception e) {
			return null;
		}
	}

	static private immutable wstring MENUGROUPNAME = "<Menu>";
	/**
	History:
		1.00β18:
			新規追加。
	*/
	static Text menuGroupName() {
		return (cast(wstring)MENUGROUPNAME).toText;
	}

}

class Sub: GroupBase {
	private Main MainGroup;
	///
	this(Main MainGroup, Kareha SubGroup)
	in {
		assert(MainGroup);
	}
	body {
		super(SubGroup);
		this.MainGroup = MainGroup;
	}

	Main mainGroup() {
		return MainGroup;
	}

	bool haveItems(Text SubGroupName) {
		if(SubGroupName.toString in ks[0]) {
			auto k=ks[0][SubGroupName.toString];
			return k !is null && k.tsvLength;
		}

		return false;
	}
}

debug(group) import std.string, nemuxi.file.items.item;
debug(group) unittest {
	auto aki=new AkiDocument("
Group$test1	test2	test5	test4	test3	test6
Group/test1$c	a	b	aaa
Group/test2$A	B
Group/test3$あ	い	う
Group/test4$い	ろ	は
Group/test5$X	Y	Z
Group/test1/a$item-1-1-1	item-1-1-2	item-1-1-3
Group/test1/b$item-1-2-1	item-1-2-2	item-1-2-3
Group/test1/c$ie	<NEMUXI>	URI
	".splitlines);
	auto main=new Main(aki["Group"]);
	assert(main.baseLength == 6);
	
	assert(main.baseName(0) == "test1");
	assert(main.baseName(1) == "test2");
	assert(main.baseName(2) == "test5");
	assert(main.baseName(3) == "test4");
	assert(main.baseName(4) == "test3");
	assert(main.baseName(5) == "test6");
	
	assert(main.baseNames[0] == main.baseName(0));
	assert(main.baseNames[1] == main.baseName(1));
	assert(main.baseNames[2] == main.baseName(2));
	assert(main.baseNames[3] == main.baseName(3));
	assert(main.baseNames[4] == main.baseName(4));
	assert(main.baseNames[5] == main.baseName(5));

	assert(main.inBaseName(Text("test1")));

	assert(main.childLength(0) == 4);
	assert(main.childNames(0)[0] == "c");
	assert(main.childNames(0)[1] == "a");
	assert(main.childNames(0)[2] == "b");
	assert(main.childNames(0)[3] == "aaa");
	
	assert(main.childLength(4) == 3);
	assert(main.childNames(4)[0] == "あ");
	assert(main.childNames(4)[1] == "い");
	assert(main.childNames(4)[2] == "う");
	
	assert(main.childLength(Text("test2")) == 2);
	assert(main.childNames(Text("test2"))[0] == "A");
	assert(main.childNames(Text("test2"))[1] == "B");

	auto sub = main.subGroup(Text("test1"));
	assert(sub.baseLength == 4);
	assert(sub.haveItems(Text("a")));
	assert(!sub.haveItems(Text("aaa")));
}
/**
指定サブグループからボタン型・メニュー型アイテムの一覧を取得。

久しぶりに見たら引数大混乱。

Params:
	ItemIDs = アイテムのID。

	ItemTree = ItemID[n]が格納されたデータ。

	ShowIcon = アイコンを取得するか。

	IconSize = アイコンの大きさ。

Return:
	どばー。
*/
BM_ITEM[] ButtonMenuGroupItems(in Text[] ItemIDs, Kareha ItemTree, bool ShowIcon, ICONSIZE IconSize)
in {
	assert(ItemTree);
}
body {
	BM_ITEM[] BMItems;
	foreach(ItemName; ItemIDs) {
		if(ItemName.toString in ItemTree) {
			Item item=new Item(ItemName, ItemTree);
			BMItems ~= ItemToButtonMenuItem(item.type() != Item.TYPE.LINK ? item: new LinkItem(item, ItemTree), ShowIcon, IconSize);
		}
	}

	return BMItems;
}


