﻿/**
文字列。

かなーりstring⇔Text変換の飛び交うﾈﾑぃの心臓部。
*/
module nemuxi.system.text;

debug import std.stdio: wl = writefln, pl = printf;
debug(text) void main() {}

import win32.core;

import std.conv;
import std.utf;
import std.string;
import std.ctype;
//import std.c.wcharh;

import std.uni;
//import std.md5;
import std.contracts;

/**
とりあえずWindowsW文字列を扱いやすく、それでいていい加減な文字列。

構造体だけど参照という謎仕様。
外部(Windows)に渡すような関数ではこれが大抵引数。

--------------------------------------------------
Text str;
str = "aaa";
assert(str == "aaa"c);
assert(str == "aaa"w);
assert(str == "aaa"d);
--------------------------------------------------

もっかしたら今後最後に0が入って.ptrでtext.ptrを渡すかも。

Bugs:
	連想配列が出来ない…。
	書き換え不可にはどうすればいいんだろう…。
*/
struct Text {
	/// 実態。
	wchar[] text;
	
	/// 空文字列の作成。
	static final Text emptyText()
	out(r) {
		assert(!r.length());
	}
	body {
		return Text("");
	}
	debug(text) unittest {
		assert(!Text.emptyText.length);
	}

	/**
	Windows用文字列の生成。

	Return:
		textの0終端文字列を生成。
		誰かに渡さないとGCの対象になるんで注意。
	*/
	wchar* ptr() const {
		return (text.dup ~ cast(wchar)0).ptr;
	}
	debug(text) unittest {
		Text t;
		t.text = "123"w.dup;
		auto p=t.ptr;
		assert(p != t.text.ptr);
		size_t i;
		while(p[i++] != '\0') {};
		assert(t.text.length + 1 == i);
	}
	
	/**
	文字列の長さを取得。
	UTF-16が一文字純正だったら素敵。

	Return:
		文字列の長さ。
	*/
	nothrow size_t length() const {
		return text.length;
	}
	debug(text) unittest {
		Text t;
		t.text = "123456"w.dup;
		assert(t.length == "123456"w.length);
		t.text = "あいうえお"w.dup;
		assert(t.length == 5);
	}
	/**
	文字列長の設定。

	Params:
		n = 設定する文字列長。
	*/
	void length(size_t n) {
		text.length = n;
	}
	debug(text) unittest {
		Text t;
		t.text = "123"w.dup;
		t.length = 1;
		assert(t.text == "1");
	}
	/**
	文字列のコピー。

	Return:
		コピーされた文字列。
	*/
	Text dup() const {
		Text t;
		t.text = text.dup;
		return t;
	}
	debug(text) unittest {
		Text t;
		t.text = "abc"w.dup;
		Text t2=t.dup;
		assert(t.text !is t2.text);
		assert(t.text == t2.text);
	}
	/**
	文字列のinvariantなコピー。

	Return:
		Dのwstring。
	*/
	wstring idup() const {
		return text.idup;
	}
	/// ソート。
	Text sort() {
		text.sort;
		return this;
	}
	/// ditto
	Text reverse() {
		text.reverse;
		return this;
	}
	/**
	死んでしまえ。
	*/
	const string toString() {
		return toUTF8(text);
	}
	/+
	/// UTF-8文字列。
	string text8() const {
		return toUTF8(text);
	}
	+/
	deprecated alias toString text8;
	/// UTF-16文字列。
	//alias dup text16;
	/// UTF-32文字列。
	dstring text32() const {
		return std.utf.toUTF32(text);
	}

	const(char*) toStringz() const {
		return std.string.toStringz(toString);
	}
	/**
	History:
		1.00β11, 2009/08/24:
			wchar*がnullの場合に正しく処理するように修正。
	*/
	Text opAssign(T)(in T arg) {
		static if(is(T == Text)) {
			text = cast(wchar[])arg.text;
		} else static if(is(T == wchar*) || is(T == const(wchar*))) {
			if(arg !is null) {
				//text = to!(wchar[])(arg[0..wcslen(arg)]);
				text = to!(wstring)(arg[0..wcslen(arg)]).dup;
			} else {
				text = null;
			}
		} else {
			//text = to!(wchar[])(arg);
			text = (to!(wstring)(cast(T)arg)).dup;
		}
		return this;
	}
	debug(text) unittest {
		Text t;
		t = "123"c.idup;
		assert(t.text == "123");
		t = "456"w.idup;
		assert(t.text == "456");
		t = "789"d.idup;
		assert(t.text == "789");
		t = "abc"c.dup;
		assert(t.text == "abc");
		t = "def"w.dup;
		assert(t.text == "def");
		t = "ghi"d.dup;
		assert(t.text == "ghi");

		t = ubyte.max;
		assert(t.text == "255");
		t = short.max;
		assert(t.text == "32767");

		t = 1.5;
		assert(t.text == "1.5");

		t = new Object();
		assert(t.text == toUTF16((new Object()).toString()));

		t = x"313233003435360037383900"w.dup.ptr;
		assert(t.text == "123");

	}

	/// ~
	Text opCat(T)(in T arg) const {
		static if(is(T == Text)) {
			return Text(text ~ arg.text);
		} else {
			return Text(text ~ to!(wstring)(arg).dup);
		}
	}
	debug(text) unittest {
		//assert((Text(123) ~ "456") == "123456");
		Text t;
		t = "123";
		assert((t ~ "456").text == "123456");
		assert((t ~ 456).text == "123456");
		assert((t ~ 4.5).text == "1234.5");
		Text t2;
		t2 = 456;
		assert((t ~ t2).text == "123456");
	}
	
	/// ~=
	Text opCatAssign(T)(T arg) {
		//text ~= to!(wchar[])(arg);
		text ~= Text(arg).text;
		return this;
	}
	debug(text) unittest {
		Text t;
		t.text = "aaa"w.dup;
		t ~= 9.5;
		assert(t == "aaa9.5");
		
	}

	/**
	--------------------------------
	auto t = Text("test");
	assert(t.text == "test"w)
	--------------------------------
	*/
	static Text opCall(T)(T arg) {
		Text t;
		t = arg;

		return t;
	}
	debug(text) unittest {
		auto text=new wchar[100];
		text[0]='a';
		for(auto i='A', j=0; i < 'Z'+1; i++, j++)
		text[j] = i;
		text[26]=0;
		auto t=Text(text.ptr);
		assert(t == text[0..26], t.toString);
		auto a=Text(132);
		assert(a==132);
		assert(Text(123) == 123);
		assert(Text(1234) == Text(123) ~ 4);
		assert(a == Text(132));
		assert(a == Text(`132`));
		assert(a == Text(`13`) ~ 2);
		assert(a == Text(`13`) ~ `2`);
		assert(a == Text(`13`) ~ '2');
		auto b=Text(2);
		assert(a == (Text(`13`) ~ b));
		//assert(a == Text(`13`) ~ Text(2));
	}

	version(none) debug(text) unittest {
		Text[Text] a;

		Text aaa=`123`;
		Text bbb=`456`;

		a[aaa] = bbb;
	}

	/// [n]
	wchar opIndex(size_t n) const {
		return text[n];
	}
	/// [n] = value
	wchar opIndexAssign(wchar value, size_t n) {
		return text[n] = value;
	}
	/// []
	Text opSlice() {
		return this;//Text(text[]);
	}
	/// [n..m]
	const Text opSlice(in size_t start, in size_t end)
	in {
		assert(start < end, format("%s[%s..%s]", text, start, end));
	}
	body {
		Text t;
		t.text = text[start..end].dup;
		return t;
	}
	debug(text) unittest {
		Text t;
		t = "LICENSE";
		assert(t[0..1]=="L");
		assert(t[0..2]=="LI");
		assert(Text(t.text[0..3]) == "LIC");
		assert(Text(t.text[0..7]) == "LICENSE");
	}

	/// ==
	bool opEquals(T)(in T arg) const {
		static if(is(T == Text)) {
			return text == arg.text;
		} else {
			return text == to!(wstring)(arg);
		}
	}
	/// >= > < <=
	int opCmp(T)(in T arg) const {
		static if(is(T == Text) || is(T == const(Text))) {
			return cmp(text, arg.text);
		} else {
			return cmp(text, to!(wstring)(arg));
		}
	}

	/**
	History:
		1.00β11, 2009/08/24:
			新規追加。
	*/
	const int cmpi(in Text text) {
		return lstrcmpi(ptr, text.ptr);
	}


	// Text.text半直接操作-------------------------------------------------
	///
	const Text chomp() {
		/+
		auto s = std.string.chomp(text);
		text = s;
		return this;
		+/
		return Text(std.string.chomp(text));
	}
	debug(text) unittest {
		Text t="abc\r\n";
		assert(t.chomp == "abc", t.toString());
	}
	///
	Text stripl() {
		if(!text.length) return this;
		size_t i;
		while(isspace(text[i])) i++;
		//text = text[i..$];
		//return this;
		return Text(text[i..$]);
	}
	debug(text) unittest {
		Text t=" abc";
		assert(t.stripl == "abc", t.toString());
		t = "                              abc";
		assert(t.stripl == "abc", t.toString());
		t = "\t\t\t\t\tabc";
		assert(t.stripl == "abc", t.toString());
		t = "\r\r\r\r\rabc";
		assert(t.stripl == "abc", t.toString());
		t = "\n\n\n\n\nabc";
		assert(t.stripl == "abc", t.toString());
	}
	///
	Text stripr() {
		if(!text.length) return this;
		size_t i=text.length-1;
		while(isspace(text[i])) i--;
		//text = text[0..i+1];
		//text.length = i+1;
		//return this;
		return Text(text[0..i+1]);
	}
	debug(text) unittest {
		Text t="abc ";
		assert(t.stripr == "abc", t.toString());
		t = "abc                              ";
		assert(t.stripr == "abc", t.toString());
		t = "abc\t\t\t\t\t";
		assert(t.stripr == "abc", t.toString());
		t = "abc\r\r\r\r\r";
		assert(t.stripr == "abc", t.toString());
		t = "abc\n\n\n\n\n";
		assert(t.stripr == "abc", t.toString());
		auto tt=t.stripr;
	}
	///
	Text strip() {
		return this.stripl.stripr;
	}
	debug(text) unittest {
		Text t="      abc      ";
		assert(t.strip == "abc", t.toString());
	}
	private const int Find(in wchar c, in bool ul, in bool Head) {
		auto temp=text.dup;
		if(!Head) {
			temp = text.dup.reverse;
		}
		foreach(i, t; temp) {
			if(ul && isUniAlpha(c) && isUniAlpha(t)) {
				if(toUniUpper(c) == toUniUpper(t)) {
					return i;
				}
			} else if(c == t) {
				return i;
			}
		}
		return -1;
	}
	
	const int find(in wchar c) {
		return Find(c, false, true);
	}
	const int ifind(in wchar c) {
		return Find(c, true, true);
	}
	const int rfind(in wchar c) {
		return Find(c, false, false);
	}
	const int irfind(in wchar c) {
		return Find(c, true, false);
	}
	debug(text) unittest {
		Text t="123456789abcdABCD";
		assert(t.find('1') == 0);
		assert(t.find('9') == 8);
		assert(t.find('a') == 9);
		assert(t.find('A') == 13);
		
		assert(t.ifind('1') == 0);
		assert(t.ifind('9') == 8);
		assert(t.ifind('a') == 9);
		assert(t.ifind('A') == 9);
		
		assert(t.rfind('D') == 0);
		assert(t.irfind('d') == 0);
	}

	const size_t count(in wchar c) {
		size_t Counter=0;
		
		foreach(t; text) {
			if(t == c) {
				Counter++;
			}
		}

		return Counter;
	}
	debug(text) unittest {
		Text t;
		t="abc;d;e;;f;g";
		assert(t.count(';') == 5);
		t="";
		assert(t.count(';') == 0);
		t=";";
		assert(t.count(';') == 1);
	}
	const Text[] split(in wchar c) {
		auto Count=count(c);
		if(!Count) {
			return null;
		}
		auto list = new Text[Count];
		for(auto i=0, j=0, k=0; i < text.length; i++) {
			if(text[i] == c) {
				list[k++] = text[j..i];
				j = i + 1;
			}
		}
		return list;
	}
	
	
	/***/
	const Text quot(T)(T arg) {
		return quot(arg, arg);
	}
	const Text quot(T)(T left, T right) {
		return Text(Text(left) ~ text ~ Text(right));
	}
	debug(text) unittest {
		Text t="abc";
		assert(t.quot("'") == `'abc'`);
	}
/+
.empty もう要素がない時に true を返す 
.next Range の左端をひとつ右に動かす 
.retreat Range の右端をひとつ左に動かす 
.head Range の最左端の要素を返す 
.toe Range の最右端の要素を返す 
+/




	
}

Text toText(in string s) {
	return Text(s);
}
Text toText(in wstring s) {
	return Text(s);
}
Text toText(in dstring s) {
	return Text(s);
}

/**
Text/string配列作成。

見返すと使用頻度が多いコードだったんで作成。
*/
Text[] Texts(T...)(in T args)
in {
	assert(args.length);
}
body {
	auto ts=new Text[args.length];
	
	foreach(i, arg; args) {
		ts[i] = arg;
	}

	return ts;
}
debug(text) unittest {
	auto t=Texts(1, 2, "3", '4', 5.0, "");
	assert(t[0] == "1");
	assert(t[1] == "2");
	assert(t[2] == "3");
	assert(t[3] == "4");
	assert(t[4] == "5");
	assert(t[5] == "");
}
/// ditto
Text[] Texts(T)(in T[] args) {
	if(!args.length) {
		return null;
	}
	
	auto ts=new Text[args.length];
	
	for(auto i=0; i < ts.length; i++) {
		ts[i] = args[i];
	}

	return ts;
}
/// ditto
string[] Strings(in Text[] texts)
in {
	assert(texts.length);
}
body {
	auto ss=new string[texts.length];

	for(auto i=0; i < ss.length; i++) {
		ss[i] = texts[i].toString();
	}

	return ss;
}
debug(text) unittest {
	Text[] ts=[
		Text("1"),
		Text("2"),
		Text("3")
	];
	assert(Strings(ts) == ["1", "2", "3"]);
}
debug(text) unittest {
	Text t=0;
	wl("%s",t);
}

