﻿/**
RAIIをやりやすく。

気が付けば解放保証人みたいな扱い。
そしてそれすらにもなっていないことに気付く。
*/
module nemuxi.system.raii;

debug import std.stdio: wl = writefln, pl = printf;
debug(raii) void main() {}

import std.string;

public import win32.winnt;

import nemuxi.base;

/**
RAII基盤

Killメソッドを実装すれば後は自殺するだけさ。

History:
	1.00β11:
		使用していなかったログ機能の撤廃。
*/
abstract class Raii {
	/**
	デスタラクタ時に自殺を行うフラグ。
	*/
	protected bool Suicide;

	this(bool Suicide) {
		this.Suicide = Suicide;
	}

	/**
	自殺内容。
	Suicideの内容は影響しない。

	Exception:
		失敗時に例外を投げる。
	*/
	abstract protected void Kill();

	/**
	手軽に自殺。
	呼び出した後にRaii.Suicideはfalseに設定される。
	Suicideの内容は判定されない。

	Return:
		自殺と判定された場合はtrue、自殺に見せかけた他殺とか自殺未遂ならfalseを返す。
	*/
	final bool kill() {
		try {
			Kill();
			Suicide = false;
			return true;
		} catch(Exception e) {
			return false;
		}
	}

	/// さぁ逝こう！
	final ~this() {
		if(Suicide) {
			Kill();
		}
	}

	/**
	身辺整理フラグ。

	Return:
		準備できているならtrue、まだならfalseを返す。
	*/
	final const bool suicide() {
		return Suicide;
	}
	/+
	bool suicide(bool flag) {
		return Suicide = flag;
	}
	+/

	/+
	override string toString() {
		return format(
			"<%s> %s",
			super.toHash(),
			super.toString
		);
	}
	+/
}
/**
Raiiを強制。
*/
interface IRaii {
	/**
	自殺内容。
	Suicideの内容は影響しない。

	Exception:
		失敗時に例外を投げる。
	*/
	protected void Kill();
}


debug {
	import std.string;
	struct _RAII_ {
		size_t Total   = 0;
		size_t Success = 0;
		size_t Failure = 0;

		string toString() {
			return format(
				"KillResource >> ALL: %08s -> OK: %08s, NG: %08s, THROUGH: %08s",
				Total,
				Success,
				Failure,
				Total - (Success-Failure)
			);
		}
	}
	_RAII_ _raii_;
	static ~this() {
		if(_raii_.Total) {
			wl("%s", _raii_);
			Logger.write(_raii_);
		}
	}
}
/**
リソースハンドル削除用テンプレート。
限られた用途に限り有効。

Params:
	ResourceName = リソース名＠直接は関係なし。

	ResourceHandle = 解放すべきリソースハンドル。

	KillFunc = ResourceHandleを引数として受け取る処理。
	           成功すると(未実装のKillFuncFailで判定)ResourceHandleはResourceHandle.initが代入される。

	KillFuncFail = KillFuncの失敗値。
*/
template KillResource(alias ResourceName, alias ResourceHandle, alias KillFunc, alias KillFuncFailValue) {
	import std.string;
	override void Kill() {
		debug _raii_.Total++;
		if(ResourceHandle) {
			if(KillFunc(ResourceHandle) != KillFuncFailValue) {
				debug _raii_.Success++;
				ResourceHandle = ResourceHandle.init;
			} else {
				debug _raii_.Failure++;
				throw new NemuxiException(format(this.classinfo.name ~ ".RAII失敗(%s()[%0x8]", ResourceName, toHash()));
			}
		}
	}
}


/**
ハンドルオブジェクト専用RAII。

Windowsで大活躍のHANDLEをサポートするけど作ったのがnemuxi.drawを作った後だったので足枷。
*/
abstract class HandleRaii: Raii {
	/// ハンドル。
	protected HANDLE Handle;

	/**
	コンストラクタ。

	Params:
		Handle = 設定するハンドル。

		Suicide = 死亡フラグ。
	*/
	this(HANDLE Handle, bool Suicide) {
		this.Handle  = Handle;
		super(Suicide);
	}

	/// ハンドルの取得。
	final const HANDLE opCall() {
		return Handle;
	}
}
/**
HandleRaiiのコンストラクタ統一。

----------------------------------------
class SuperHandle: HandleRaii, IRaii {
	mixin HandleRaiiClass;
	
	override void Kill() {
		del(Handle);
	}
}
class ExHandle: SuperHandle {
	mixin HandleRaiiClass;
}

auto ex = ExHandle(handle, true);
----------------------------------------
*/
template HandleRaiiClass() {
	override this(HANDLE Handle, bool Suicide) {
		super(Handle, Suicide);
	}
	/+
	override HANDLE opCall() {
		return super.opCall();
	}
	+/
}

/**
初期化が必要。
*/
interface IInitialize {
	/// 初期化の実行。
	void initialize();
}

