﻿/**
アイテム関数。
*/
module nemuxi.file.items.itemfunc;

debug import std.stdio: wl = writefln, pl = printf;

debug(itemfunc) void main() {}

import std.string;
import std.conv;
import std.file;
import std.path;

import win32.windef;

import etc.kareki.kareki;

import nemuxi.base;
import nemuxi.file.file;
import nemuxi.file.items.item;
import nemuxi.file.items.linkitem;
public import nemuxi.system.timer;
import nemuxi.image.icon;

/**
*/
class ItemDateTime: DateTime {
	invariant() {
		assert(Mask == MASK.DATE+MASK.TIME);
		assert(0 <= SystemTime.wYear && SystemTime.wYear <= 9999);
		assert(Format    == FORMAT);
		assert(MonthList == MONTHLIST);
	}

	invariant FORMAT    = "[YYYY]/[M]/[DD] [hh]:[mm]:[ss]";
	invariant MONTHLIST = ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"];
	invariant OUTSTRING = "YYYY/MM/DD HH:MM:SS";
	
	/**
	*/
	this() {
		super();
		Format    = FORMAT;
		MonthList = MONTHLIST;
	}
	/**
	*/
	this(WORD Year, WORD Month, WORD Day, WORD Hour, WORD Minute, WORD Second, WORD MilliSecond) {
		this();
		
		year   = Year;
		month  = Month;
		day    = Day;
		hour   = Hour;
		minute = Minute;
		second = Second;
		ms     = MilliSecond;
		ChangeWeek();
	}
	/**
	ItemException
	*/
	this(string date) {
		this();
		
		if(date.length != OUTSTRING.length) {
			throw new ItemException("内部保持用日時情報長が不正", EC.ITEM_DATE_LENGTH);
		}

		try {
			auto time=split(date);
			auto ymd=split(time[0], "/");
			auto hms=split(time[1], ":");
			
			if(ymd.length + hms.length != 6)
				throw new ItemException("日時情報分割数が不正。", EC.ITEM_DATE_SPLIT);
			
			year   = to!(WORD)(ymd[0]);
			month  = to!(WORD)(ymd[1]);
			day    = to!(WORD)(ymd[2]);
			hour   = to!(WORD)(hms[0]);
			minute = to!(WORD)(hms[1]);
			second = to!(WORD)(hms[2]);

			bool check=true;
			check &= year > 0 && year < 10000;
			check &= month >= 1 && month <= 12;
			check &= day >= 1 && day <= 31;
			check &= hour >= 0 && hour <= 23;
			check &= minute >= 0 && minute <= 59;
			check &= second >= 0 && second <= 59;
			if(!check)
				throw new ItemException("日時情報がへんてこりん", EC.ITEM_DATE_INFO);
			ChangeWeek();

		} catch(Exception e) {
			throw new ItemException("日時情報変換時に例外", EC.ITEM_DATE, e);
		}
	}
	
	override void year(in WORD Year) {
		if(0 <= Year && Year <= 9999) {
			super.year(Year);
		} else {
			throw new ItemException("年情報が不正", EC.ITEM_DATE);
		}
	}
	override const WORD year()
	out(r) {
		assert(0 <= r && r <= 9999);
	}
	body {
		return super.year();
	}
}


/+
/**
数値データから内部保持用文字列へ変換。

常識的な値であることが強制される。

Params:
	date = 年月日時分秒情報。

Return:
	YYYY/MM/DD HH:MM:SS
*/
string DateToString(ItemDateTime date)
in {
	assert(date.year > 0 && date.year < 10000); // やったー、9999年まで保障されてるよ！
	assert(date.month >= 1 && date.month <= 12);
	assert(date.day >= 1 && date.day <= 31);
	assert(date.hour >= 0 && date.hour <= 23);
	assert(date.minute >= 0 && date.minute <= 59);
	assert(date.second >= 0 && date.second <= 59);
}
body {
	return format(
		"%04s/%02s/%02s %02s:%02s:%02s",
		date.year,
		date.month,
		date.day,
		date.hour,
		date.minute,
		date.second
	);
}
/**
内部保持用文字列から数値データへ変換。

常識的な値であることが強制される。

Params:
	date = YYYY/MM/DD HH:MM:SS。

Return:
	変換された日時情報。

Exception:
	ミスればNemuxiExceptionを投げる。
*/
ItemDateTime StringToDate(string date)
out(r) {
	assert(r.year > 0 && r.year < 10000); // やったー、9999年まで保障されてるよ！
	assert(r.month >= 1 && r.month <= 12);
	assert(r.day >= 1 && r.day <= 31);
	assert(r.hour >= 0 && r.hour <= 23);
	assert(r.minute >= 0 && r.minute <= 59);
	assert(r.second >= 0 && r.second <= 59);
}
body {
	if(date.length != "YYYY/MM/DD HH:MM:SS".length)
		throw new NemuxiException("内部保持用日時情報長が不正");


	auto value=new ItemDateTime();
	with(value) try {
		auto time=split(date);
		auto ymd=split(time[0], "/");
		auto hms=split(time[1], ":");
		
		if(ymd.length + hms.length != 6)
			throw new NemuxiException("日時情報分割数が不正。");
		
		year   = to!(WORD)(ymd[0]);
		month  = to!(WORD)(ymd[1]);
		day    = to!(WORD)(ymd[2]);
		hour   = to!(WORD)(hms[0]);
		minute = to!(WORD)(hms[1]);
		second = to!(WORD)(hms[2]);

		bool check=true;
		check &= year > 0 && year < 10000;
		check &= month >= 1 && month <= 12;
		check &= day >= 1 && day <= 31;
		check &= hour >= 0 && hour <= 23;
		check &= minute >= 0 && minute <= 59;
		check &= second >= 0 && second <= 59;
		if(!check)
			throw new NemuxiException("日時情報がへんてこりん");

	} catch(Exception e) {
		throw new NemuxiException("日時情報変換時に例外", EC.ITEM_DATE, e);
	}
	
	return value;
}
+/
debug(itemfunc) unittest {
	ItemDateTime date1=new ItemDateTime();
	ItemDateTime date2;
	with(date1) {
		year   = 2009;
		month  = 1;
		day    = 1;
		hour   = 0;
		minute = 0;
		second = 0;
	}
	assert(date1.toString == "2009/01/01 00:00:00", date1.toString);
	date2 = new ItemDateTime("2009/01/01 00:00:00");
	assert(date1.toString == date2.toString);
	with(date1) {
		year   = 1;
		month  = 12;
		day    = 31;
		hour   = 23;
		minute = 59;
		second = 59;
	}
	assert(date1.toString == "0001/12/31 23:59:59");
	date2 = new ItemDateTime("0001/12/31 23:59:59");
	assert(date1.toString == date2.toString);

	try {
		new ItemDateTime("005/112/31 23:59:59");
		assert(false);
	} catch(NemuxiException) {
		assert(true);
	}
}

string GetName(Item item)
in {
	assert(item);
}
body {
	return item.name;
}

	private string PlusFolderSplit(string path) {
		if(path.length && path[$-1] != '\\') {
			return path ~ '\\';
		}
		return path;
	}

/**
親フォルダの取得。

Params:
	item = 親フォルダを取得するアイテム。
	       <table>
	         <thead>
	           <tr>
	             <th>Item.TYPE</th>
	             <th>フォルダ</th>
	           </tr>
	         </thead>
	         <tbody>
	           <tr>
	             <th>NORMAL</th>
	             <td>親フォルダ</td>
	           </tr>
	           <tr>
	             <th>URI</th>
	             <td>
	               iten.uriが有効なローカルアドレスならその親フォルダ。
	               無効な場合はスペース分割("が存在する場合は括弧として扱う)した第一要素をローカルアドレスとする。
	               分割要素が無効であれば作業フォルダを使用、それも無理ならnullを返す。
	             </td>
	           </tr>
	           <tr>
	             <th>MULTI</th>
	             <td>
	               作業フォルダを使用。
	               作業フォルダが有効でなければnull。
	             </td>
	           </tr>
	           <tr>
	             <th>LINK</th>
	             <td></td>
	           </tr>
	         </tbody>
	       </table>
*/
string GetFolder(Item item, lazy Kareha ItemBase=null)
in {
	assert(item);
}
out(r) {
	if(r)
	assert(r[$-1]=='\\', r);
}
body {
	string GetDirName(string path) {
		auto Folder=dirname(path);
		if(Folder[$-1] != '\\') {
			Folder ~= '\\';
		}
		return Folder;
	}
	
	with(Item.TYPE) switch(item.type()) {
		case NORMAL: {
			return GetDirName(item.address);
		}
		case URI: {
			try {
				auto Folder = PlusFolderSplit(item.workFolder);
				
				if(isdir(Folder)) {
					return Folder;
				} else {
					return GetDirName(Folder);
				}
			} catch(std.file.FileException e) {
				if(exists(item.uri)) {
					return GetDirName(item.uri);
				} else if(item.uri.length) {
					char   SplitChar;
					size_t StartIndex;
					if(item.uri[0] == '"') {
						SplitChar  = '"';
						StartIndex = 1;
					} else {
						SplitChar  = ' ';
						StartIndex = 0;
					}
					for(auto i=StartIndex; i < item.uri.length; i++) {
						if(item.uri[i] == SplitChar) {
							auto Folder=GetDirName(item.uri[StartIndex..i]);
							if(exists(Folder)) {
								return Folder;
							}
							break;
						}
					}
				}
			}
			
			return null;
		}
		case MULTI: {
			auto Folder = PlusFolderSplit(item.workFolder());
			/+
			if(Folder.length && Folder[$-1] != '\\') {
				Folder ~= '\\';
			}
			+/
			
			try {
				return isdir(Folder) != 0
					? Folder
					: null
				;
			} catch(std.file.FileException e) {
				return null;
			}
		}
		case LINK: {
			auto link=new LinkItem(item, ItemBase);
			return GetFolder(link, ItemBase);
		}
		default:
			assert(false);
	}
}
debug(itemfunc) unittest {
	auto aki=newItem();
	auto item=new Item("ie", aki[`Items`]);
}

string GetWorkFolder(Item item, lazy Kareha ItemBase=null)
in {
	assert(item);
}
out(r) {
	if(r)
	assert(r[$-1]=='\\', r);
}
body {
	with(Item.TYPE) switch(item.type()) {
		case NORMAL:
			if(item.workFolder.length) {
				return PlusFolderSplit(item.workFolder);
			} else {
				auto Folder=dirname(item.address);
				if(Folder.length) {
					return PlusFolderSplit(Folder);
				}
				
				return null;
			}
		case URI, MULTI:
			if(item.workFolder.length) {
				return PlusFolderSplit(item.workFolder);
			} else {
				return null;
			}
		case LINK:
			return GetWorkFolder(new LinkItem(item, ItemBase), ItemBase);
		default:
			assert(false);
	}
}

Icon GetIcon(Item item, ICONSIZE IconSize, lazy Kareha ItemBase=null)
in {
	assert(item);
}
body {
	Text IconAddress=void;
	if(item.type() != Item.TYPE.LINK)
	IconAddress=item.iconAddress();
	
	switch(item.type()) {
		case Item.TYPE.LINK:
			return GetIcon(new LinkItem(item, ItemBase), IconSize, ItemBase);
		case Item.TYPE.NORMAL:
			if(!FILE.isExistence(IconAddress)) {
				Text FileAddress=item.address();
				if(FILE.isExistence(FileAddress)) {
					IconAddress = FileAddress;
				} else {
					IconAddress = Text.emptyText();
				}
			}
			break;
		case Item.TYPE.URI, Item.TYPE.MULTI:
			if(!FILE.isExistence(IconAddress)) {
				IconAddress = Text.emptyText();
			}
			break;
		default:
			assert(false);
	}
	if(IconAddress.length) {
		return GetFileIcon(IconAddress, item.iconIndex(), IconSize);
	} else {
		SYSICON SysIcon;
		if(item.type() == Item.TYPE.NORMAL) {
			SysIcon = SYSICON.ASTERISK;
		} else {
			assert(item.type() != Item.TYPE.LINK);
			SysIcon = item.type() == Item.TYPE.URI ? SYSICON.URI:SYSICON.MULTI;
		}
		return GetSystemIcon(SysIcon, IconSize);
	}
}

/**
循環参照チェック。

Params:
	item = 調べたいリンク/マルチアイテム。

	ItemRoot = 基底アイテムツリー。

	MaxCall = 呼び出し最大数。

Exception:
	循環参照していた場合にNemuxiExceptionを投げる。

Bugs:
	itemからの呼び出し回数制限。
	とりあえずこれで回避。
*/
void ReferenceCheck(Item item, Kareha ItemBase, size_t MaxCall = 1)
in {
	assert(item);
	assert(item.type() == Item.TYPE.LINK || item.type() == Item.TYPE.MULTI);
	assert(MaxCall > 0);
	assert(ItemBase);
}
body {
	// 自身のIDを登録
	size_t[string] List=[item.itemID: 1];

	void Refs(Item RefItem)
	in {
		assert(RefItem);
		assert(RefItem.type() == Item.TYPE.LINK || RefItem.type() == Item.TYPE.MULTI);
	}
	body {
		void ItemCheck(string ItemID, Item TargetItem) {
			if(TargetItem.type() == Item.TYPE.LINK || TargetItem.type() == Item.TYPE.MULTI) {
				List[ItemID] += 1;
				if(List[ItemID] > MaxCall) {
					throw new NemuxiException(ItemID);
				}
				Refs(TargetItem);
			}
		}
		
		Item TempItem=null;
		if(RefItem.type() == Item.TYPE.LINK) {
			TempItem = new LinkItem(RefItem, ItemBase);
			ItemCheck(RefItem.link, TempItem);
			/+
			if(TempItem.type() == Item.TYPE.LINK || TempItem.type() == Item.TYPE.MULTI) {
				List[RefItem.link] += 1;
				if(List[TempItem.link] > 1) {
					throw new NemuxiException(TempItem.itemID);
				}
				Refs(TempItem);
			}
			+/
		} else {
			assert(RefItem.type() == Item.TYPE.MULTI);
			auto multis=RefItem.multi();
			foreach(ItemID; multis) {
				TempItem = new Item(ItemID, ItemBase);
				ItemCheck(TempItem.itemID, TempItem);
				/+
				if(TempItem.type() == Item.TYPE.LINK || TempItem.type() == Item.TYPE.MULTI) {
					List[TempItem.itemID] += 1;
					if(List[TempItem.itemID] > 1) {
						throw new NemuxiException(TempItem.itemID);
					}
					Refs(TempItem);
				}
				+/
			}
		}
	}

	try {
		Refs(item);
	} catch(Exception e) {
		throw new NemuxiException(format("循環参照, %s", List), EC.NONE, e);
	}
}
debug(itemfunc) unittest {
	auto aki=newItem();
	auto item=new Item("MULTI", aki[`Items`]);

	if(item.type() == Item.TYPE.LINK || item.type() == Item.TYPE.MULTI)
	ReferenceCheck(item, aki[`Items`], 10);
}
/+
/***/
string OptionToPlain(string Option) {
	replace
}
+/

