﻿/**
ファイルとパス。

Phobosがchar[]大好きだったんで必要分だけ再構築。
*/
module nemuxi.file.file;

debug import std.stdio: wl = writefln, pl = printf;
debug(file) void main() {}

import std.contracts;
import std.file;
public import std.file: FileException;
import std.path;
import std.conv;

import win32.windows;
import win32.shlwapi;

import nemuxi.base;
/+
class NemuxiFileException: NemuxiException {
	mixin NemuxiExceptionClass;
}
+/

static struct PATH {
	static:

	bool isFolderPath(in Text path) {
		return path[path.length-1] == '\\';
	}
	debug(file) unittest {
		assert(isFolderPath(`asd\asd\`w.toText));
		assert(!isFolderPath(`asd\asd`c.toText));
	}
	Text addFolderSep(in Text path) {
		if(isFolderPath(path)) {
			return cast(Text)path;
		}
		return path ~ '\\';
	}
	debug(file) unittest {
		assert(addFolderSep(Text(`asd\asd\`)) == `asd\asd\`);
		assert(addFolderSep(Text(`asd\asd`)) == `asd\asd\`);
	}
	Text delFolderSep(in Text path) {
		if(auto len=path.length) {
			while(isFolderPath(path[0..len])) {
				if(len--) continue; else break;
			}
			return path[0..len];
		} else {
			return Text.emptyText();
		}
	}
	debug(file) unittest {
		assert(delFolderSep(Text(`asd\asd\`)) == `asd\asd`);
		assert(delFolderSep(Text(`asd\asd`)) == `asd\asd`);
		assert(delFolderSep(Text(``)) == ``);
		assert(delFolderSep(Text(`a`)) == `a`);
		assert(delFolderSep(Text(`a\\\`)) == `a`);
		assert(delFolderSep(Text(`LICENSE`)) == `LICENSE`);
	}

	/**
	拡張子取得。

	Params:
		path = 拡張子を取得したいパス。

	Return:
		拡張子。
	*/
	Text getExtension(in Text path) {
		/+
		auto Head=path.ptr;
		auto Dot=PathFindExtension(Head);
		return Text(Dot+1);
		+/
		auto p=PathFindExtension(path.ptr);
		if(p) {
			return Text(cast(wchar*)(p+1));
		} else {
			return Text.emptyText;
		}
	}
	debug(file) unittest {
		assert(getExtension(Text(``)) == ``);
		assert(getExtension(Text(`123456`)) == ``);
		assert(getExtension(Text(`123.56`)) == `56`);
		assert(getExtension(Text(`.23.56`)) == `56`);
		assert(getExtension(Text(`.23456`)) == `23456`);
		assert(getExtension(Text(`abc\def`)) == Text(``));
		assert(getExtension(Text(`abc\.def`)) == Text(`def`));
		Text t;
		assert(getExtension(t) == Text(``));
	}
	/**
	ファイル名取得。

	Params:
		path = ファイル名を取得したいパス。

	Return:
		ファイル名。
		こいつは拡張子付き。
	*/
	Text getFileName(in Text path) {
		return Text(PathFindFileName(path.ptr));
	}
	debug(file) unittest {
		assert(getFileName(Text(``)) == ``);
		assert(getFileName(Text(`Z:\aaa.txt`)) == `aaa.txt`);
		assert(getFileName(Text(`Z:\aaatxt`)) == `aaatxt`);
		assert(getFileName(Text(`Z:\aaatxt\`)) == `aaatxt\`);
		assert(getFileName(Text(`LICENSE`)) == `LICENSE`);
	}
	/**
	ファイル名取得。

	Params:
		path = ファイル名を取得したいパス。

	Return:
		ファイル名。
		こいつは拡張子無し。

	History:
		1.00β11:
			スライスで死ぬ対策。
	*/
	Text getName(in Text path) {
		auto name=delFolderSep(getFileName(path));
		
		auto ext =getExtension(path);
		
		if(ext.length && name.length > ext.length+1) {
			return name[0..name.length - ext.length-1];
		} else {
			return name;
		}
	}
	debug(file) unittest {
		assert(getName(Text(``)) == ``);
		assert(getName(Text(`.`)) == `.`);
		assert(getName(Text(`.aaa`)) == ``);
		assert(getName(Text(`Z:\aaa.txt`)) == `aaa`);
		assert(getName(Text(`Z:\aaatxt\`)) == `aaatxt`);
		assert(getName(Text(`aaabbb`)) == `aaabbb`);
		assert(getName(Text(`aaa.bbb`)) == `aaa`);
		assert(getName(Text(`aaa.bbb.ccc`)) == `aaa.bbb`);
	}
	/**
	親フォルダ取得。

	Params:
		親フォルダを取得したいパス。

	Return:
		親フォルダ。
	*/
	Text getOwnerFolder(in Text path) {
		return addFolderSep(Text(dirname(delFolderSep(path).text)));
	}
	debug(file) unittest {
		assert(getOwnerFolder(Text(``)) == `.\`);
		assert(getOwnerFolder(Text(`123`)) == `.\`);
		assert(getOwnerFolder(Text(`123\456`)) == `123\`);
		assert(getOwnerFolder(Text(`123\456\`)) == `123\`);
		assert(getOwnerFolder(Text(`123\456\789`)) == `123\456\`);
		assert(getOwnerFolder(Text(`C:\`)) == `C:\`);
	}

	/**
	ファイルパス比較。

	単純にファイル文字列の比較であって相対パス等は考慮しない。
	
	History:
		1.00β11:
			新規追加。
	*/
	int cmp(in Text path1, in Text path2) {
		return path1.cmpi(path2);
		//return lstrcmpi(path1.ptr, path2.ptr);
	}

	/**
	History:
		1.00β11:
			nemuxi.baseから移動。
	*/
	private Text MyPath() {
		// 最悪だ
		debug(nemuxi) {
			auto text=new wchar[MY_MAX_PATH];

			if(!GetModuleFileName(GetModuleHandle(null), text.ptr, MY_MAX_PATH)) {
				throw new NemuxiException("自分の居場所さえ分からない", EC.NEMUXI_MYPATH);
			}
			
			return Text(text.ptr);
		} else debug {
			return Text(`Z:\nemuxi\nemuxi\nemuxi.exe`);
		} else {
			auto text=new wchar[MY_MAX_PATH];

			if(!GetModuleFileName(GetModuleHandle(null), text.ptr, MY_MAX_PATH)) {
				throw new NemuxiException("自分の居場所さえ分からない", EC.NEMUXI_MYPATH);
			}
			
			return Text(text.ptr);
		}
	}
	/**
	自身のアドレス取得。
	
	History:
		1.00β11:
			新規作成。
	*/
	Text myPath() {
		return MyPath;
	}
	/**
	自身の親フォルダ。
	
	History:
		1.00β11:
			nemuxi.baseから移動。
	*/
	Text myFolder() {
		return getOwnerFolder(myPath);
	}
}


static struct FILE {
	static:

	enum ATTRIBUTE {
		NONE                = 0xFFFFFFFF,
		ARCHIVE             = FILE_ATTRIBUTE_ARCHIVE,             /// アーカイブファイルまたはアーカイブディレクトリです。アプリケーションはこの属性を、ファイルのバックアップや削除のためのマークとして使います。
		COMPRESSED          = FILE_ATTRIBUTE_COMPRESSED,          /// 指定されたファイルまたはディレクトリは圧縮されています。ファイルの場合、ファイル内の全データが圧縮されていることを意味します。ディレクトリの場合、そのディレクトリ内に新しく作成されるファイルまたはサブディレクトリが、既定で圧縮状態になることを意味します。
		DEVICE              = FILE_ATTRIBUTE_DEVICE,              /// 予約済み。使わないでください。
		DIRECTORY           = FILE_ATTRIBUTE_DIRECTORY,           /// 指定されたハンドルは、ディレクトリに関連しています。
		ENCRYPTED           = FILE_ATTRIBUTE_ENCRYPTED,           /// 指定されたファイルまたはディレクトリは暗号化されています。ファイルの場合、ファイル内の全データストリームが暗号化されていることを意味します。ディレクトリの場合、そのディレクトリ内に新しく作成されるファイルまたはサブディレクトリが、既定で暗号化状態になることを意味します。
		HIDDEN              = FILE_ATTRIBUTE_HIDDEN,              /// 隠しファイルまたは隠しディレクトリです。通常のディレクトリリスティングでは表示されません。
		NORMAL              = FILE_ATTRIBUTE_NORMAL,              /// 指定されたファイルまたはディレクトリには、特に属性はありません。単独で返った場合にのみ、この属性は有効です。
		NOT_CONTENT_INDEXED = FILE_ATTRIBUTE_NOT_CONTENT_INDEXED, /// Windows 2000：このファイルは、「インデックスサービス」の対象になっていません。
		OFFLINE             = FILE_ATTRIBUTE_OFFLINE,             /// Windows 2000：このファイルのデータは、すぐには利用できません。この属性は、ファイルのデータがオフラインの記憶装置へ物理的に移動されたことを示します。この属性は、 Windows 2000 の階層記憶管理ソフトウェアである「リモート記憶域」が利用するものです。アプリケーションは、任意にこの属性を変更するべきではありません。
		READONLY            = FILE_ATTRIBUTE_READONLY,            /// このファイルまたはディレクトリは読み取り専用です。アプリケーションはこのファイルの読み取りを行えますが、書き込みや削除はできません。ディレクトリの場合、アプリケーションは削除を行えません。
		REPARSE_POINT       = FILE_ATTRIBUTE_REPARSE_POINT,       /// このファイルには、再解析ポイントが関連付けられています。
		SPARSE_FILE         = FILE_ATTRIBUTE_SPARSE_FILE,         /// このファイルは、スパースファイル（疎なファイル、未使用の領域が多い、または同じ値が長く続くファイル）です。
		SYSTEM              = FILE_ATTRIBUTE_SYSTEM,              /// このファイルまたはディレクトリは、オペレーティングシステムの一部、またはオペレーティングシステム専用です。
		TEMPORARY           = FILE_ATTRIBUTE_TEMPORARY,           /// このファイルは、一時ファイルとして使われています。ファイルシステムは、データをハードディスクのような大容量記憶装置へ書き込む代わりに、高速なアクセスが行えるよう、すべてのデータをメモリ内に維持することを試みます。アプリケーションは、必要がなくなった段階で一時ファイルをすぐに削除するべきです。
	}
	ATTRIBUTE getFileAttributes(in Text path) {
		return cast(ATTRIBUTE)GetFileAttributes(path.ptr);
	}
	
	/**
	指定アドレスは存在するか。

	Params:
		path = 調べたいアドレス。

	Return:
		存在すればtrue、しなければfalse。
	*/
	bool isExistence(in Text path) {
		return getFileAttributes(path) != ATTRIBUTE.NONE;
	}
	/**
	指定アドレスはフォルダか。

	Params:
		path = 調べたいアドレス。

	Return:
		フォルダならtrue。

	Exception:
		アドレス自体が存在しなければNemuxiFileException。
	*/
	bool isFolder(in Text path) {
		enforce(isExistence(path), new NemuxiFileException("ファイル無い。"));
		return (getFileAttributes(path) & FILE_ATTRIBUTE_DIRECTORY) != 0;
	}
	/**
	指定アドレスはファイルか。

	Params:
		path = 調べたいアドレス。

	Return:
		ファイルならtrue。

	Exception:
		アドレス自体が存在しなければNemuxiFileException。
	*/
	bool isFile(in Text path) {
		return !isFolder(path);
	}
	/**
	フォルダ作成。

	Params:
		path = フォルダ
	*/
	bool makeFolder(in Text path, bool Recurse=true) {
		if(Recurse) {
			const left=PATH.getOwnerFolder(path);
			isExistence(left) || makeFolder(left, true);
			return makeFolder(path, false);
		} else {
			return cast(bool)CreateDirectory(PATH.delFolderSep(path).ptr, null);
		}
	}


}
/+
/**
*/
bool IsPath(in Text path) {
	return exists(path.toString);
}
/**
*/
bool IsFile(in Text path) {
	return isfile(path.toString);
}
/**
*/
bool IsFolder(in Text path) {
	return isdir(path.toString);
}

/**
*/
bool MakeFolder(in Text path) {
	try {
		mkdirRecurse(path.toString);
		return true;
	} catch(Exception e) {
		return false;
	}
}

bool IsFolderPath(in Text path) {
	return path[path.length-1] == '\\';
}

/**
*/
Text GetExt(in Text path) {
	return Text(getExt(path.toString));
}
/**
*/
Text GetName(in Text path) {
	return Text(getName(path.toString));
}
/**
*/
Text BaseName(in Text path) {
	return Text(basename(path.toString));
}
/**
*/
Text FolderName(in Text path) {
	if(path[path.length-1] == '\\') {
		return PathAddSep(Text(dirname(path[0..path.length-1].toString)));
	} else {
		return PathAddSep(Text(dirname(path.toString)));
	}
}
+/
// ----------------------------------------------------------------------------
/+
/**
*/
Text PathAddSep(in Text path)
in {
	assert(path.length);
}
body {
	if(path[path.length-1] == '\\') {
		return cast(Text)path;
	}
	return path ~ '\\';
}
+/

Text GetNameEx(in Text path) {
	Text Name;
	/+
	if(IsPath(path)) {
		
	} else {
	}
	+/
	
	if(path.length) {
		Name = PATH.getOwnerFolder(path);
	}
	if(!Name.length) {
		Name = PATH.getOwnerFolder(path[0..path.length-1]);
		if(!Name.length) {
			if(FILE.isExistence(path)) {
				// ドライブ？
				Name = path;
			} else {
				Name = path;
			}
		}
	}
	
	return Name;
}





