﻿/**

*/
module nemuxi.file.exec;

debug import std.stdio: wl = writefln, pl = printf;

debug(exec) void main() {}

import std.string;
import std.contracts;
import core.thread;

import win32.windows;

import nemuxi.base;
import nemuxi.file.file;
import nemuxi.negui.negui;

/**
ファイル実行。

Params:
	FileAddress = 実行するファイル。
	
	Option = FileAddressに渡すオプション。

	WorkFolder = 作業フォルダ。
	             無効なものであればFileAddressの親フォルダを使用。

	Show = 実行時の表示フラグ。

	OpenExplorer = FileAddressがフォルダの場合にtrueならエクスプローラで、falseなら普通に開く。
	
	hWnd = 実行ファイルの親ウィンドウ。

Retrun:

Exception:
	FileAddressが存在しなければNemuxiExceptionを投げる。

History:
	1.00β11:
		フォルダ判定部分がファイル存在判定になっていた部分を修正。
*/
HINSTANCE FileExecute(
	in Text FileAddress,
	in Text Option,
	in Text WorkFolder,
	in SHOW Show,
	in bool OpenExplorer=false,
	in NeGui gui=null
)
in {
	assert(FileAddress.length());
//	assert(Show != SHOW.NONE);
}
body {
	//HINSTANCE ret=void;
	SHELLEXECUTEINFO Shell;
	try {
		Shell.cbSize = Shell.sizeof;
		Shell.hwnd   = gui ? gui(): GetDesktopWindow();
		Shell.fMask  = SEE_MASK_DOENVSUBST | SEE_MASK_FLAG_NO_UI;
		if(!FILE.isExistence(FileAddress)) {
			auto Spliter=FileAddress.find(' ');
			Text OptionEx;
			if(Spliter != -1) {
				OptionEx = FileAddress[Spliter+1..FileAddress.length];
			} else {
				Spliter = FileAddress.length;
			}
			// path
			///+
			auto ret = ShellExecute(
				Shell.hwnd,
				null,
				FileAddress[0..Spliter].ptr,
				(OptionEx ~ Option).ptr,
				null,
				Show
			);
			if(ret > cast(HANDLE)32) {
				return ret;
			} else {
				Err.code = cast(uint)ret;
				throw new NemuxiFileException(
					format("%s[%s]", Err.toString(), FileAddress),
					EC.FILE_EXECUTE_PATH
				);
			}
			//+/
			/+
			Shell.lpFile = FileAddress.ptr;
			Shell.nShow  = Show;
			Shell.lpVerb = Text("open").ptr;
			+/
		} else if(FILE.isFile(FileAddress)) {
			// ファイルの場合
			bool Wark=void;
			if(WorkFolder.length) {
				try {
					Wark = FILE.isFolder(WorkFolder);
				} catch(FileException e) {
					//Logger.write(e);
					Wark = false;
				}
			} else {
				Wark = false;
			}
			/+
			ret = ShellExecute(
				hWnd,
				null,
				FileAddress.ptr,
				Option.ptr,
				Wark ? WorkFolder.ptr: FolderName(FileAddress).ptr,
				Show
			);
			+/
			Shell.lpFile       = FileAddress.ptr;
			Shell.lpParameters = Option.ptr;
			Shell.lpDirectory  = Wark ? WorkFolder.ptr: PATH.getOwnerFolder(FileAddress).ptr,
			Shell.nShow        = Show;
			Shell.lpVerb       = Text("open").ptr;
		} else {
			// フォルダの場合
			/+
			ret = ShellExecute(
				hWnd,
				OpenExplorer ? Text("explore").ptr: Text("open").ptr,
				FileAddress.ptr,
				null,
				null,
				Show
			);
			+/
			Shell.lpFile = FileAddress.ptr;
			Shell.lpVerb = OpenExplorer ? Text("explore").ptr: Text("open").ptr;
			Shell.nShow  = Show;
		}
		Err.init;
		enforce(
			ShellExecuteEx(&Shell),
			new NemuxiFileException(format("%s [%s]", Err.toString(), FileAddress.toString), EC.FILE_EXECUTE_PROPERTY)
		);
	} catch(Exception e) {
		throw new NemuxiException("ファイル実行例外", EC.FILE, e);
	}

	return Shell.hInstApp;
}
debug(exec) unittest {
	//FileExecute(Text(`C:\Program Files\Opera\Opera.exe`), Text(`http://www.usamimi.info/~wokonomono/`), Text.emptyText, NeGui.SW.SHOW);
}

/// フォルダを開く方法。
enum FOLDER {
	NORMAL,
	EXPLORER,
	PROGRAM,
}
/**
フォルダオープン。

内部でFileExecuteを呼んでいる。

Params:
	FolderAddress = フォルダのアドレス。
	                x\y\z\って感じで最後が\のものがﾈﾑぃではフォルダ認定される。

	OpenState = どのようにしてフォルダを開くか。

	Show = 実行時の表示フラグ。

	OpenProgram = OpenStateがFOLDER.PROGRAMの時にフォルダを開くプログラムを指定。
	              FOLDER.PROGRAMの場合はnullだと表明違反。
	              こいつの後ろにFolderAddressがそのまんま引っ付くのでこれを呼ぶまでに加工すべし。
*/
HINSTANCE FolderExecute(
	in Text FolderAddress,
	in FOLDER OpenState,
	in SHOW Show,
	in Text* OpenProgram=null
)
in {
	assert(FolderAddress.length);
	if(OpenState == FOLDER.PROGRAM)
		assert(OpenProgram);
//	assert(Show != SHOW.NONE);
}
body {
	try {
		if(OpenState != FOLDER.PROGRAM) {
			//FolderAddress = PathAddSep(FolderAddress);

			return FileExecute(
				PATH.addFolderSep(FolderAddress),
				Text.emptyText(),
				Text.emptyText(),
				Show,
				OpenState == FOLDER.EXPLORER
			);
		} else {
			enforce(
				FolderAddress.length,
				new NemuxiFileException("肝心のフォルダパスが完全に無効", EC.FILE_EXECUTE_FOLDER)
			);
		}
		return FileExecute(
			*OpenProgram,
			FolderAddress,
			FolderAddress,
			Show
		);
	} catch(Exception e) {
		throw new NemuxiException("フォルダ実行無効, " ~ FolderAddress.toString, EC.FILE, e);
	}
}
/+
Function ShowProperty(pAddress As BytePtr) As HINSTANCE
	Dim sExe As SHELLEXECUTEINFO

	With sExe
		.cbSize = SizeOf(SHELLEXECUTEINFO)
		.hWnd   = GetDesktopWindow()
		.nShow  = SW_SHOW
		.fMask  = SEE_MASK_NOCLOSEPROCESS Or SEE_MASK_INVOKEIDLIST Or SEE_MASK_FLAG_NO_UI
		.lpVerb = "properties"
		.lpFile = pAddress

		ShellExecuteEx(sExe)

		Return sExe.hInstApp
	End With
End Function
+/
/**
ABから移行、例外追加。
*/
HINSTANCE PropertyExecute(in Text FileAddress, in NeGui gui=null) {
	SHELLEXECUTEINFO Shell;

	with(Shell) {
		cbSize = Shell.sizeof;
		hwnd   = gui ? gui(): GetDesktopWindow();
		nShow  = SW_SHOWNORMAL;
		fMask  = SEE_MASK_NOCLOSEPROCESS | SEE_MASK_INVOKEIDLIST | SEE_MASK_FLAG_NO_UI;
		lpVerb = Text("properties").ptr;
		lpFile = FileAddress.ptr;
	}
	Err.init;
	enforce(
		ShellExecuteEx(&Shell),
		new NemuxiFileException(format("%s[%s]", Err.toString(), FileAddress.toString), EC.FILE_EXECUTE_PROPERTY)
	);

	return Shell.hInstApp;
}

/**
Windowsじゃshell使えないっぽいのでそれっぽいのを実装。

Exception:
	実行したときに非常に変だったらNemuxiExceptionを投げる。
*/
class ShellExec: Thread {
	private {
		Text CommandLine;
		DWORD ReturnCode;
	}
	DWORD returnCode() {
		return ReturnCode;
	}
	Text ShellCode;

	this(in Text CommandLine) {
		this.CommandLine = CommandLine;
		super(&run);
	}

	private void run() {
		SECURITY_ATTRIBUTES SecAtributes=void;
		with(SecAtributes) {
			nLength = SecAtributes.sizeof;
			lpSecurityDescriptor = null;
			bInheritHandle = true;
		}

		HANDLE hRead, hWrite;
		
		enforce(CreatePipe(&hRead, &hWrite, &SecAtributes, 0), new NemuxiException(Err.toString(), EC.PROCCESS));

		STARTUPINFO StartUpInfo;
		with(StartUpInfo) {
			cb          = StartUpInfo.sizeof;
			dwFlags     = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
			wShowWindow = SW_SHOWDEFAULT;
			hStdInput   = hRead;
			hStdOutput  = hWrite;
			hStdError   = hWrite;
		}
		PROCESS_INFORMATION ProccessInfo;
		enforce(
			// 引数多いな
			CreateProcess(
				null,            // 実行可能モジュールの名前
				CommandLine.ptr, // コマンドラインの文字列
				null,            // セキュリティ記述子
				null,            // セキュリティ記述子
				true,            // ハンドルの継承オプション
				0,               // 作成のフラグ
				null,            // 新しい環境ブロック
				null,            // カレントディレクトリの名前
				&StartUpInfo,    // スタートアップ情報
				&ProccessInfo    // プロセス情報
			),
			new NemuxiException(CommandLine.toString() ~ ":" ~ Err.toString(), EC.PROCCESS_CREATE)
		);
		// 落ち着くまで待機
		WaitForInputIdle(ProccessInfo.hProcess, INFINITE);
		CloseHandle(ProccessInfo.hThread);

		// 終了まで待機。
		WaitForSingleObject(ProccessInfo.hProcess, INFINITE);

		// 終了コード取得
		GetExitCodeProcess(ProccessInfo.hProcess, &ReturnCode);
		CloseHandle(ProccessInfo.hProcess);

		size_t NoPipeRead;
		do{
			size_t PipeRead;
			NoPipeRead=0;

			enforce(PeekNamedPipe(hRead, null, 0, null, &PipeRead, &NoPipeRead), new NemuxiException(Err.toString(), EC.PROCCESS));
			if(PipeRead) {
				size_t FileRead;
				auto ReadData=new char[PipeRead];
				enforce(ReadFile(hRead, ReadData.ptr, PipeRead, &FileRead, NULL), new NemuxiException(Err.toString(), EC.PROCCESS));
				if(FileRead) {
					ShellCode ~= cast(string)ReadData[0..FileRead];
				} else {
					break;
				}
			}
		} while(NoPipeRead);
	}

	override string toString() {
		return format("%s%s[CODE = %s]", ShellCode.toString, newline, ReturnCode);
	}
}
debug(exec) unittest {
	auto n=new ShellExec(Text("winver"));
	n.start;n.join;
	wl(n.toString);
}
