﻿/**
メニュー。

一応メニューバーとポップアップメニューを考慮。
まぁポップアップしか作りませんが。
*/
module nemuxi.negui.window.menu.menu;

debug import std.stdio: wl = writefln, pl = printf;
debug(menu) void main() {}

import std.string;
import std.contracts;

import win32.windows;

import nemuxi.negui.system.base;
import nemuxi.negui.negui;
import nemuxi.negui.system.type.basic;
import nemuxi.negui.system.type.enumerated;
import nemuxi.negui.draw.brush;
import nemuxi.negui.system.meta.memberproperty;

///
class MenuException: NeGuiException {
	mixin MixInNeGuiException;
}

/**
Bugs:
	MENUINFO.dwMenuDataはNeGuiで予約。
	勘弁してください。
*/
private struct MENU {
	MENUINFO MenuInfo;
	mixin(SMixInStructHiddenOriginal!(MENUINFO)(q{MenuInfo}));

	void initialize() {
		MenuInfo.cbSize = MenuInfo.sizeof;
	}


	enum MASK: size_t {
		APPLYTOSUBMENUS = MIM_APPLYTOSUBMENUS, /// 
		BACKGROUND      = MIM_BACKGROUND,      /// 
		HELPID          = MIM_HELPID,          /// 
		MAXHEIGHT       = MIM_MAXHEIGHT,       /// 
		MENUDATA        = MIM_MENUDATA,        /// 
		STYLE           = MIM_STYLE,           /// 
	}
	mixin(SMixInStructGetSet!(MASK)("mask", q{MenuInfo.fMask}));

	enum STYLE {
		AUTODISMISS = MNS_AUTODISMISS, /// 
		CHECKORBMP  = MNS_CHECKORBMP,  /// 
		DRAGDROP    = MNS_DRAGDROP,    /// 
		MODELESS    = MNS_MODELESS,    /// 
		NOCHECK     = MNS_NOCHECK,     /// 
		NOTIFYBYPOS = MNS_NOTIFYBYPOS, /// 
	}
	mixin(SMixInStructGetSet!(STYLE)("style", q{MenuInfo.dwStyle}));

	mixin(SMixInStructGetSet!(STYLE)("maxHeight", q{MenuInfo.cyMax}));

	const Brush backBrush() {
		if(MenuInfo.hbrBack) {
			return new Brush(MenuInfo.hbrBack, false);
		} else {
			return null;
		}
	}
	void backBrush(in Brush BackBrush) {
		if(BackBrush) {
			MenuInfo.hbrBack = BackBrush();
		} else {
			MenuInfo.hbrBack = null;
		}
	}
	
	mixin(SMixInStructGetSet!(void*)("data", q{MenuInfo.dwMenuData}));
}


/**
メニュー基底。

メニューハンドルをラップして使用。
アイテムは相対位置ではなくIDで指定する(※今後内部フラグで切り替えるようにする可能性有)。

History:
	1.00β20:
		[P] 各々のハンドルが自身のオブジェクトを保持。
*/
abstract class Menu: Raii, IHandle {
	/// GC用サブメニュー。
	protected Menu[COMMAND_ID] SubMenus;
	/// メニューハンドル。
	protected HMENU hMenu;

	/**
	History:
		1.00β11:
			使用していなかったログ機能の撤廃。
	*/
	override void Kill() {
		if(this.SubMenus) {
			scope Values=this.SubMenus.values;
			foreach(key; this.SubMenus.keys) {
				this.SubMenus.remove(key);
			}
			foreach(SubMenu; Values) {
				delete SubMenu;
			}
			this.SubMenus = null;
		}
		if(hMenu) {
			if(DestroyMenu(hMenu) != 0) {
				hMenu = hMenu.init;
			} else {
				throw new MenuException(this.toString.toText);
			}
		}
	}


	
	/// コンストラクタ
	this(HMENU hMenu, bool Suicide=false, bool DataSet=true)
	in {
		assert(hMenu);
	}
	body {
		super(Suicide);
		this.hMenu = hMenu;

		if(DataSet) try {
			//データ設定
			auto Menu=GetMenuInformation(hMenu);
			if(Menu.data) {
				throw new MenuException(Text("既に設定済み[OLD=(%08x), NEW=(%08x)]", Menu.data, hMenu));
			}
			Menu.data = cast(void*)this;
			SetMenuInformation(hMenu, Menu);
		} catch(MenuException e) {
			throw new MenuException(Text("データセット(%08x)", hMenu), e);
		}
	}

	version(unittest) private void ItemCheck(ref const(MENUITEM) MenuItem, string file=__FILE__, int line=__LINE__) {
		auto s=format(" [%s(%s)]", file, line);
		assert(MenuItem.id > 0, format("<Id=%s> %s", MenuItem.id, s));
	}

	/**
	メニューアイテムの挿入。

	Params:
		MenuItem = 挿入するアイテムの情報。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。
	*/
	bool insert(ref const(MENUITEM) MenuItem)
	in {
		version(unittest) ItemCheck(MenuItem);
	}
	body {
		if(((MenuItem.mask & MENUITEM.MASK.SUBMENU) == MENUITEM.MASK.SUBMENU) && MenuItem.SubMenu && Menu.isMenu(MenuItem.SubMenu())) {
			SubMenus[MenuItem.id] = cast(Menu)MenuItem.SubMenu;
		}
		
		return cast(bool)InsertMenuItem(
			hMenu,
			MenuItem.id,
			MF_BYCOMMAND,
			MenuItem.ptr
		);
	}

	/**
	メニューアイテムの設定。

	Params:
		MenuItem = 設定するアイテムの情報。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。
	*/
	bool set(ref const(MENUITEM) MenuItem)
	in {
		version(unittest) ItemCheck(MenuItem);
	}
	body {
		if(((MenuItem.mask & MENUITEM.MASK.SUBMENU) == MENUITEM.MASK.SUBMENU) && MenuItem.SubMenu && Menu.isMenu(MenuItem.SubMenu())) {
			SubMenus[MenuItem.id] = cast(Menu)MenuItem.SubMenu;
		}
		
		return cast(bool)SetMenuItemInfo(
			hMenu,
			MenuItem.id,
			MF_BYCOMMAND,
			MenuItem.ptr
		);
	}

	/**
	History:
		1.021:
			[S] 引数属性変更。
	*/
	const bool get(ref MENUITEM MenuItem) {
		return cast(bool)GetMenuItemInfo(
			hMenu,
			MenuItem.id,
			MF_BYCOMMAND,
			MenuItem.ptr
		);
	}
	
	/**
	メニューアイテムの削除。

	Params:
		Id = 削除するアイテムID。

	Return:
		成功した場合はtrue、失敗した場合はfalseを返す。

	History:
		1.010:
			[B] パラメータ順序。
	*/
	const bool del(COMMAND_ID Id) {
		return cast(bool)DeleteMenu (
			hMenu,
			Id,
			MF_BYCOMMAND
		);
	}

	override const HANDLE opCall() {
		return hMenu;
	}

	/**
	History:
		1.081:
			[S] 引数属性変更。
	*/
	bool insertText(COMMAND_ID Id, in Text text, bool Enable=true)
	in {
		assert(Id != COMMAND_ID.init);
	}
	body {
		MENUITEM MenuItem;
		MenuItem.initialize();
		with(MenuItem) {
			mask     = MASK.ID | MASK.TYPE | MASK.STATE;
			id       = Id;
			type     = TYPE.STRING;
			state    = Enable ? STATE.ENABLED: STATE.DISABLED;
		}
		MenuItem.text    = text.ptr;
		
		return insert(MenuItem);
	}
	
	bool insertSeparator(COMMAND_ID Id=COMMAND_ID.max) {
		MENUITEM MenuItem;
		MenuItem.initialize();
		with(MenuItem) {
			mask     = MASK.ID | MASK.TYPE;
			id       = Id;
			type     = TYPE.SEPARATOR;
		}
		return insert(MenuItem);
	}
	/**
	History:
		1.081:
			[S] 引数属性変更。

		1.00β11:
			事前条件のSubMenuの判定方法を変更。
	*/
	bool insertSubMenu(COMMAND_ID Id, in Text text, Menu SubMenu, bool Enable=true)
	in {
		if(SubMenu) {
			assert(isMenu(SubMenu()));
		}
	}
	body {
		MENUITEM MenuItem;
		MenuItem.initialize();
		with(MenuItem) {
			mask     = MASK.ID | MASK.TYPE | MASK.STATE | MASK.SUBMENU;
			id       = Id;
			type     = TYPE.STRING;
			state    = Enable ? STATE.ENABLED: STATE.DISABLED;
		}
		MenuItem.text    = text.ptr;
		
		MenuItem.subMenu = SubMenu;

		return insert(MenuItem);
	}

	/**
	ID取得。

	Params:
		Position = メニューアイテムの位置。

	Return:
		PositionにあるメニューアイテムのID。

	Exception:
		Positionからメニューアイテムを取得できない場合にMenuExceptionを投げる。
	
	History:
		1.021:
			[S] メソッド属性変更。
	*/
	const COMMAND_ID posToID(size_t Position) {
		MENUITEMINFO MenuItemInfo=void;
		MenuItemInfo.cbSize = MenuItemInfo.sizeof;
		MenuItemInfo.fMask = MIIM_ID;
		if(GetMenuItemInfo(hMenu, Position, MF_BYPOSITION, &MenuItemInfo)) {
			return cast(COMMAND_ID)MenuItemInfo.wID;
		}
		throw new MenuException(Text("めにゅー"));
	}

	/// 意味ね～。
	Menu getSubMenu(COMMAND_ID Id) {
		return SubMenus[Id];
	}

	/**
	ラジオチェック設定。

	Params:
		StartID = グループの先頭。
	
		EndID = グループの最後。
	
		TargetID = チェックを入れる項目。

	Exception:
		TargetIDがグループに無ければMenuException。

	Return:
		成功すればtrue、失敗すればfalse。
	*/
	bool radio(COMMAND_ID StartID, COMMAND_ID EndID, COMMAND_ID TargetID)
	in {
		assert(StartID);
		assert(EndID);
		assert(TargetID);
	}
	body {
		if(TargetID < StartID || EndID < TargetID) {
			throw new MenuException(Text("範囲外"));
		}
		return cast(bool)CheckMenuRadioItem(hMenu, StartID, EndID, TargetID, MF_BYCOMMAND);
	}
	/**
	指定IDのチェック状態を調べる。

	Params:
		Id = 調べたいID。

	Return:
		チェックしていればtrue、していなければfalse。

	Exception:
		メニュー項目を取得できなければMenuException。
	
	History:
		1.021:
			[S] メソッド属性変更。
	*/
	const bool check(COMMAND_ID Id)
	in {
		assert(Id);
	}
	body {
		MENUITEM MenuItem;
		MenuItem.initialize();
		MenuItem.mask = MENUITEM.MASK.ID | MENUITEM.MASK.STATE;
		MenuItem.id = Id;

		if(this.get(MenuItem)) {
			return (MenuItem.state & MENUITEM.STATE.CHECKED) == MENUITEM.STATE.CHECKED;
		} else {
			throw new MenuException(Text("チェックマーク取得失敗"));
		}
	}
	/**
	指定IDのチェック状態を設定。

	Params:
		Id = 調べたいID。

		Check = ON/OFF。

	Exception:
		メニュー項目を取得・設定できなければMenuException。
	*/
	void check(COMMAND_ID Id, bool Check)
	in {
		assert(Id);
	}
	body {
		MENUITEM MenuItem;
		MenuItem.initialize();
		MenuItem.mask = MENUITEM.MASK.ID | MENUITEM.MASK.STATE;
		MenuItem.id = Id;

		if(this.get(MenuItem)) {
			if(Check) {
				MenuItem.state = MenuItem.state | MENUITEM.STATE.CHECKED;
			} else {
				MenuItem.state = MenuItem.state & ~MENUITEM.STATE.CHECKED;
			}
			if(!this.set(MenuItem)) {
				throw new MenuException(Text("チェックマーク設定失敗"));
			}
		} else {
			throw new MenuException(Text("メニュー項目取得失敗"));
		}
	}

	static bool isMenu(HANDLE Handle) {
		return cast(bool)IsMenu(Handle);
	}

	MENU_FLAG enable(COMMAND_ID Id, MENU_FLAG MenuFlags) {
		auto Old=EnableMenuItem(hMenu, Id, MenuFlags | MF_BYCOMMAND);
		enforce(Old != 0xFFFFFFFF, new MenuException(Text("ID(%08x)見つからず", Id)));
		return cast(MENU_FLAG)Old;
	}

	protected {
		static ref MENU GetMenuInformation(HMENU hMenu) {
			auto Menu=new MENU;
			Menu.initialize;
			with(MENU.MASK) Menu.mask = APPLYTOSUBMENUS | BACKGROUND | HELPID | MAXHEIGHT | MENUDATA | STYLE;
			enforce(GetMenuInfo(hMenu, Menu.ptr), new MenuException(ERR.toText));

			return *Menu;
		}
		static void SetMenuInformation(HMENU hMenu, const ref MENU Menu) {
			enforce(SetMenuInfo(hMenu, Menu.ptr), new MenuException(ERR.toText));
		}
	}
	static Menu getMenuObject(HMENU hMenu) {
		if(hMenu) {
			auto MenuInfo=GetMenuInformation(hMenu);
			if(auto menu=cast(Menu)MenuInfo.data) {
				return menu;
			}
		}
		return null;
	}

}

/**
History:
	1.00β20:
		新規作成。
*/
class OtherMenu: Menu {
	this(HMENU hMenu) {
		super(hMenu, false, false);
	}
}

Menu GetMenuNeGui(HMENU hMenu) {
	if(hMenu) {
		if(auto menu=Menu.getMenuObject(hMenu)) {
			return menu;
		} else {
			return new OtherMenu(hMenu);
		}
	} else {
		return null;
	}
}

/**
メニューアイテム

History:
	1.00β20:
		[S] aliasをdeprecatedに変更。
*/
struct MENUITEM {
	invariant() {
		if(MenuItemInfo.hSubMenu) assert(SubMenu);
		else assert(!SubMenu);
	}
	/// メニューアイテムの実態だけど通常は構造体メソッドを使用。
	MENUITEMINFO MenuItemInfo;
	mixin(SMixInStructHiddenOriginal!(MENUITEMINFO)(q{MenuItemInfo}));
	private Menu SubMenu;

	/// 初期化サイズ。
	void initialize() {
		MenuItemInfo.cbSize = MenuItemInfo.sizeof;
	}

	/// どのメンバ変数を利用するかを指定します。
	static enum MASK {
		CHECKMARKS = MIIM_CHECKMARKS, /// hbmpChecked、hbmpUnchecked メンバが有効になります。
		DATA       = MIIM_DATA      , /// dwItemData メンバが有効になります。 
		ID         = MIIM_ID        , /// wID メンバが有効になります。 
		STATE      = MIIM_STATE     , /// fState メンバが有効になります。 
		SUBMENU    = MIIM_SUBMENU   , /// hSubMenu メンバが有効になります。 
		TYPE       = MIIM_TYPE      , /// fType、dwTypeData メンバが有効になります。 
	}
	mixin(SMixInStructGetSet!(MASK)("mask", q{MenuItemInfo.fMask}));

	/// メニュー項目のタイプを、以下の定数を組み合わせて指定します。
	static enum TYPE {
		BITMAP       = MFT_BITMAP       , /// ビットマップを使用してメニューを表示します。dwTypeData メンバの下位ワードでビットマップ ハンドルを指定します。cch メンバは無視されます。 
		MENUBARBREAK = MFT_MENUBARBREAK , /// メニュー項目を、新しい行に配置します。ドロップダウン メニュー、サブメニュー、ショートカット メニューの場合は行間に線が引かれます。 
		MENUBREAK    = MFT_MENUBREAK    , /// メニュー項目を、新しい行に配置します。ドロップダウン メニュー、サブメニュー、ショートカット メニューのときは、行間に線が引かれません。 
		OWNERDRAW    = MFT_OWNERDRAW    , /// メニューの描画をオーナーウィンドウに任せます。dwTypeData メンバはアプリケーション定義の32ビット値をもちます。 
		RADIOCHECK   = MFT_RADIOCHECK   , /// hbmpChecked メンバが NULL のとき、ラジオボタンのチェックマークをメニュー項目のチェックマークにします。 
		RIGHTJUSTIFY = MFT_RIGHTJUSTIFY , /// 以降のメニュー項目を右揃えで表示します。メニュー バーにのみ有効です。 
		RIGHTORDER   = MFT_RIGHTORDER   , /// 右から左へメニュー項目を表示します。アラビア語やヘブライ語を表示するために使用します。（Windows95、WindowsNT5.0以降） 
		SEPARATOR    = MFT_SEPARATOR    , /// メニュー項目がセパレータであることを示します。dwTypeData メンバ、cch メンバは無視されます。 
		STRING       = MFT_STRING       , /// 文字列を使ってメニュー項目を表示します。dwTypeData メンバには NULL で終わる文字列バッファへのポインタを指定します。cch メンバには文字列のバイト数を指定します。 
	}
	mixin(SMixInStructGetSet!(TYPE)("type", q{MenuItemInfo.fType}));

	/// メニュー項目の状態を、以下の定数を組み合わせて指定します。
	static enum STATE {
		CHECKED   = MFS_CHECKED   , /// チェック マークを表示します。 
		DEFAULT   = MFS_DEFAULT   , /// デフォルトのメニュー項目に設定します。メニュー項目は太字で表示されます。 
		DISABLED  = MFS_DISABLED  , /// メニュー項目を選択不可にします。淡色表示にはなりません。 
		ENABLED   = MFS_ENABLED   , /// メニュー項目を有効にします。 
		GRAYED    = MFS_GRAYED    , /// メニュー項目を選択不可にし、淡色表示します。 
		HILITE    = MFS_HILITE    , /// メニュー項目を強調表示します。 
		UNCHECKED = MFS_UNCHECKED , /// チェック マークを解除します。 
		UNHILITE  = MFS_UNHILITE  , /// 強調表示を解除します。 
	}
	mixin(SMixInStructGetSet!(STATE)("state", q{MenuItemInfo.fState}));

	mixin(SMixInStructGetSet!(COMMAND_ID)("id", q{MenuItemInfo.wID}));
	

	/**
	サブメニューを所持しているか。

	History:
		1.00β20:
			[S] メソッド名をisSubMenuからhaveSubMenuに変更。
			[P] 有無の判定を厳しめに。
	*/
	const bool haveSubMenu() {
		return MenuItemInfo.hSubMenu && Menu.isMenu(MenuItemInfo.hSubMenu);
	}
	/**
	サブメニュー取得・設定。

	Params:
		SubMenu = サブメニュー。

	Return:
		自殺しないPopUpメニュー。

	Exception:
		取得時にメニューが無効の場合にMenuExceptionを投げる。
	*/
	void subMenu(Menu SubMenu) {
		if(SubMenu) {
			MenuItemInfo.hSubMenu = SubMenu.hMenu;
			this.SubMenu  = SubMenu;
			assert(MenuItemInfo.fMask && MIIM_SUBMENU);
		} else {
			MenuItemInfo.hSubMenu = null;
			this.SubMenu  = null;
		}
	}

	mixin(SMixInStructGetSet!(void*)("data", q{MenuItemInfo.dwItemData}));
	
	/// タイプデータ。
	/// 大抵は文字列。
	wchar* text() {
		return MenuItemInfo.dwTypeData;
	}
	/// ditto
	void text(wchar* TypeData) {
		MenuItemInfo.dwTypeData = TypeData;
	}
	/// ditto
	wchar* text(in Text TypeData) {
		return MenuItemInfo.dwTypeData = TypeData.ptr;
	}
	
	mixin(SMixInStructGetSet!(size_t)("textLength", q{MenuItemInfo.cch}));
}



