﻿/**
NeGui。

Windows APIを薄皮一枚で繋げるラッパー。
当初はGUIのみを対象にしてたけど段々と変な方へ走り出してる。

Bugs:
	関連する識別子の統一をしたい。
	
	refだったりポインタだったり、constだったりそうじゃなかったり。
	型も属性も何の統一性も無いなー。

Note:
	<ul>
		<li>
			識別子の統一。
			※Dのスタイルとは合わないけれども。
			<table>
				<thead>
					<tr>
						<th>識別子</th>
						<th>方針</th>
					</tr>
				</thead>
				<tbody>
					<tr>
						<td>関数名</td>
						<td>各単語の先頭は大文字。ただし配列への糖衣構文は先頭小文字</td>
					</tr>
					<tr>
						<td>メソッド</td>
						<td>Dの関数の書き方</td>
					</tr>
					<tr>
						<td>戻り値が構造体</td>
						<td>4byte(size_tくらい)以下はそのまま、それより大きい場合はref。SIZEとか8くらいの大きさならそのままでもいいかも。</td>
					</tr>
					<tr>
						<td>クラス</td>
						<td>各単語の先頭を大文字</td>
					</tr>
					<tr>
						<td>インターフェイス</td>
						<td>多重継承時にクラスなのかインターフェイスなのか明確にするため先頭に「I」、各単語の先頭を大文字</td>
					</tr>
					<tr>
						<td>構造体・共用体</td>
						<td>全部大文字でアンダースコア '_' は使用しない</td>
					</tr>
					<tr>
						<td>構造体・共用体メンバ</td>
						<td>メソッドと同じ</td>
					</tr>
					<tr>
						<td>クラス・構造体・共用体のprivate, protectedメンバ</td>
						<td>各単語の先頭を大文字</td>
					</tr>
					<tr>
						<td>定数・列挙体・列挙型メンバ</td>
						<td>全部大文字</td>
					</tr>
					<tr>
						<td>Windows API - 構造体</td>
						<td>頻繁に使うもので定数とかあるやつはラップ</td>
					</tr>
				</tbody>
			</table>
		</li>
		<li>
			GUIに関係なさげなWindows APIもOK。
		</li>
		<li>
			PhobosでできることはPhobosで。
		</li>
		<li>
			内部でメモリを拡張したりするものはその値を指定可能に。
		</li>
	</ul>
	
History:
	1.062:
		[S] deprecated修正。

*/
module nemuxi.negui.negui;

debug import std.stdio: wl = writefln, pl = printf;
debug(negui) void main() {}

//import std.string;
import std.contracts;

import win32.winuser;
import win32.core;
import win32.shellapi;
import std.contracts;


//public import nemuxi.base;
public import nemuxi.negui.system.base;
import nemuxi.negui.system.meta.memberproperty;
import nemuxi.negui.system.meta.guimethod;
import nemuxi.negui.system.meta.guistyle;
public import nemuxi.negui.draw.icon;
public import nemuxi.negui.draw.font;
public import nemuxi.negui.draw.canvas;
public import nemuxi.negui.system.type.basic;
public import nemuxi.negui.system.type.enumerated;
import nemuxi.negui.input.keyboard.keyboard;
import nemuxi.negui.layout.layout;
public import nemuxi.negui.system.type.record;
public import nemuxi.negui.event.event;
import nemuxi.negui.system.error;
import nemuxi.negui.system.literal;


/**
History:
	1.022:
		[S] 名称変更。
	
	1.021:
		[S] メンバ名変更。

	1.00β15:
		[P] sizeメンバ追加。拡張コンボボックス対策。
		[P] pointメンバ追加。size追加した整合性のため。
*/
struct NEGUIINFO {
	DWORD   exStyle;   /// 拡張ウィンドウスタイル。
	Text    className; /// ウィンドウクラス名。
	DWORD   style;     /// ウィンドウスタイル。
	NeGui   owner;     /// 親。
	ITEM_ID id;
	bool    eventLoop=false;

	POINT point; /// 初期位置。
	SIZE  size;  /// 初期サイズ。

	const Text toText() {
		return Text(
			"[NEGUIINFO]"
			"  className = %s", className,
			", exStyle = %08x", exStyle,
			", style = %08x",   style,
			", owner = %s",     owner,
			", id = %s",        id,
			", eventLoop = %s", eventLoop,
			", point(x = %s, y = %s)", point.x, point.y,
			", size(cx = %s, cy = %s)", size.cx, size.cy
		);
	}
}
deprecated alias NEGUIINFO GUIINFO;

/**
GUIオブジェクト操作用ラッパ。

取得関係のメソッドは内部がどうなっていようとconstとして処理する。
*/
abstract class NeGui {
	/**
	操作ハンドル。
	History:
		1.060:
			[S] やっぱ名前はhWndだよね。
	*/
	protected HANDLE hWnd;
	/// コンストラクタ。
	protected this(HWND hWnd)
	in {
		assert(hWnd);
	}
	body {
		this.hWnd = hWnd;
	}
	/**
	GWLP_USERDATA代わり。

	History:
		1.021:
			[P] ユーザー領域の変更。
	*/
	union {
		Object userObject; /// 
		void*  userData;   /// 
	}

	/**
	GUI部分生成。

	やっとこさthisと分離。

	Exception:
		ウィンドウハンドルが無効なときにNeGuiException。
	
	History:
		1.032:
			[B] エラーコードが無視されてたっぽい。
			[S] 識別子合わせ。
	
		1.00β15:
			新規作成。
			[P] 拡張コンボボックスのリストサイズが変になるっぽいので適当に初期値設定。
	*/
	protected HWND MakeNeGui(const ref NEGUIINFO NeGuiInfo) {
		auto hWnd=CreateWindowEx(
			NeGuiInfo.exStyle,
			NeGuiInfo.className.ptr,
			NeGuiInfo.className.ptr,
			NeGuiInfo.style,
			NeGuiInfo.point.x, NeGuiInfo.point.y,
			NeGuiInfo.size.cx, NeGuiInfo.size.cy,
			NeGuiInfo.owner ? NeGuiInfo.owner(): GetDesktopWindow(),
			cast(HANDLE)NeGuiInfo.id,
			GetModuleHandle(null),
			NeGuiInfo.eventLoop ? cast(void*)this: null
		);
		auto code=ERR.code;
		if(!hWnd) {
			throw new NeGuiException(Text("Gui生成失敗"),
				new NeGuiException(ERR.toText(code), new NeGuiException(NeGuiInfo.toText)));
		}

		return hWnd;
	}

	this(const ref NEGUIINFO NeGuiInfo) {
		this.hWnd = MakeNeGui(NeGuiInfo);
		this.guiObject(this);
	}
	
	/// 操作ハンドル取得。
	final HWND opCall() const {
		return hWnd;
	}
	override final bool opEquals(Object obj) {
		auto gui=cast(NeGui)obj;
		return this == gui.hWnd;
	}
	final bool opEquals(HANDLE Handle) const {
		return hWnd == Handle;
	}

	/// 意味を成さないsend/post引数。
	enum INT NONE=0;
	/**
	SendMessageラッパ。

	Bugs:
		やっちゃいけないんだろうけどconst属性。
	*/
	final const LRESULT send(UINT Message, WPARAM wParam, LPARAM lParam) {
		return SendMessage(hWnd, Message, wParam, lParam);
	}
	/**
	PostMessageラッパ。
	
	Bugs:
		sendと同じく。
	*/
	final const LRESULT post(UINT Message, WPARAM wParam, LPARAM lParam) {
		return PostMessage(hWnd, Message, wParam, lParam);
	}
	
	/// アイテム情報取得フラグ。
	static enum GWL {
		EXSTYLE    = GWL_EXSTYLE,     /// 拡張ウィンドウスタイルを取得します。
		STYLE      = GWL_STYLE,       /// ウィンドウスタイルを取得します。
		WNDPROC    = GWLP_WNDPROC,    /// ウィンドウプロシージャへのポインタ、またはウィンドウプロシージャへのポインタを表すハンドルを取得します。ウィンドウプロシージャを呼び出すには、CallWindowProc 関数を使わなければなりません。
		HINSTANCE  = GWLP_HINSTANCE,  /// アプリケーションインスタンスのハンドルを取得します。
		HWNDPARENT = GWLP_HWNDPARENT, /// 親ウィンドウがある場合、そのハンドルを取得します。
		ID         = GWLP_ID,         /// ウィンドウ ID を取得します。
		USERDATA   = GWLP_USERDATA,   /// ウィンドウに関連付けられた 32 ビット値を取得します。この 32 ビット値は、ウィンドウを作成したアプリケーションで使用する目的で各ウィンドウが持っているものです。この値の初期値は 0 です。
	}
	/// Get/SetWindowLongPtrラッパ。
	protected final LONG_PTR GetItemInfo(GWL Index) {
		return GetWindowLongPtr(hWnd, Index);
	}
	/// ditto
	protected final LONG_PTR SetItemInfo(GWL Index, LONG_PTR NewInfo) {
		return SetWindowLongPtr(hWnd, Index, NewInfo);
	}
	/// ditto
	final const LONG_PTR getItemInfo(int Index) {
		return GetWindowLongPtr(hWnd, Index);
	}
	/// ditto
	final LONG_PTR setItemInfo(int Index, LONG_PTR NewInfo) {
		return SetWindowLongPtr(hWnd, Index, NewInfo);
	}
	final const LONG_PTR exStyle() {
		return this.getItemInfo(GWL.EXSTYLE);
	}
	final LONG_PTR exStyle(LONG_PTR NewInfo) {
		return this.setItemInfo(GWL.EXSTYLE, NewInfo);
	}
	final const LONG_PTR style() {
		return this.getItemInfo(GWL.STYLE);
	}
	final LONG_PTR style(LONG_PTR NewInfo) {
		return this.setItemInfo(GWL.STYLE, NewInfo);
	}
	final const WNDPROC proc() {
		return cast(WNDPROC)this.getItemInfo(NeGui.GWL.WNDPROC);
	}
	final WNDPROC proc(WNDPROC NewProc) {
		return cast(WNDPROC)this.setItemInfo(NeGui.GWL.WNDPROC, cast(LONG_PTR)NewProc);
	}
	/**
	History:
		1.021:
			[P] 固定値。
	*/
	final const Text className(size_t CLASSNAME_BUFFER=BUFFER.TEXT) {
		//immutable CLASSLEN = 256;

		/+
		auto name = new wchar[CLASSLEN];
		
		if(!GetClassName(hItem, name.ptr, CLASSLEN)) {
			throw new NeGuiException(ERR.toText);
		}
		+/
		auto name = new wchar[CLASSNAME_BUFFER];

		enforce(GetClassName(hWnd, name.ptr, name.length), new NeGuiException(ERR.toText));

		return Text(name.ptr);
	}
	/// ditto
	final const NeGui guiObject() {
		return cast(NeGui)cast(void*)getItemInfo(GWL.USERDATA);
	}
	final NeGui guiObject(NeGui NewGuiData) {
		return cast(NeGui)cast(void*)setItemInfo(GWL.USERDATA, cast(LONG_PTR)cast(void*)NewGuiData);
	}

	///
	LayoutManager layoutManager;
	
	/**

	Bugs:
		取れなかった場合にnullを返すようにせねば。

	*/
	static NeGui getGuiObject(HWND hWnd) {
		return cast(NeGui)cast(void*)GetWindowLongPtr(hWnd, GWL_USERDATA);
	}

	mixin(SMixInItemExStyleGetSet("doubleFrame", q{WS_EX_DLGMODALFRAME}, false));
	mixin(SMixInItemExStyleGetSet("clientEdge",  q{WS_EX_CLIENTEDGE},    false));
	mixin(SMixInItemExStyleGetSet("transParent", q{WS_EX_TRANSPARENT},   false));
	mixin(SMixInItemExStyleGetSet("windowEdge",  q{WS_EX_WINDOWEDGE},    false));
	mixin(SMixInItemExStyleGetSet("staticEdge",  q{WS_EX_STATICEDGE},    false));
	
	/// WS_BORDER, @reLoad
	mixin(SMixInItemStyleGetSet("border",      q{WS_BORDER}, false));
	mixin(SMixInItemStyleGetSet("frameDialog", q{WS_DLGFRAME}, false));
	deprecated alias frameDialog dialogFrame;
	
	mixin(SMixInItemStyleGetSet("hScroll", q{WS_HSCROLL}, false));
	mixin(SMixInItemStyleGetSet("vScroll", q{WS_VSCROLL}, false));

	
	/// アイテムクラス情報取得フラグ。
	static enum CLASSFLAG {
		ATOM          = GCW_ATOM,            /// ウィンドウクラスを一意的に識別するアトム値を取得します。これは、RegisterClassEx 関数が返すアトムと同じです。
		CBCLSEXTRA    = GCL_CBCLSEXTRA,      /// クラスに関連付けられた拡張クラスメモリのサイズをバイト単位で取得します。
		CBWNDEXTRA    = GCL_CBWNDEXTRA,      /// ウィンドウに関連付けられた拡張ウィンドウメモリのサイズをバイト単位で取得します。このメモリへのアクセス方法については、GetWindowLongPtr 関数の説明を参照してください。
		HBRBACKGROUND = GCL_HBRBACKGROUND,  /// クラスに関連付けられた背景ブラシのハンドルを取得します。
		HCURSOR       = GCL_HCURSOR,        /// クラスに関連付けられたカーソルのハンドルを取得します。
		HICON         = GCL_HICON,          /// クラスに関連付けられたアイコンのハンドルを取得します。
		HICONSM       = GCL_HICONSM,        /// クラスに関連付けられた小さいアイコンのハンドルを取得します。
		HMODULE       = GCL_HMODULE,        /// クラスを登録したモジュールのハンドルを取得します。
		MENUNAME      = GCL_MENUNAME,       /// クラスに関連付けられたメニューリソースを識別するメニュー名文字列へのポインタを取得します。
		STYLE         = GCL_STYLE,           /// ウィンドウクラスのスタイルビットを取得します。
		WNDPROC       = GCL_WNDPROC,        /// ウィンドウプロシージャのアドレス、またはウィンドウプロシージャのアドレスを表すハンドルを取得します。ウィンドウプロシージャを呼び出すには、CallWindowProc 関数を使わなければなりません。
	}
	deprecated alias CLASSFLAG GCL;
	version(none) {
		/// GetClassLongPtrラッパ。
		ULONG_PTR getClassInfo(CLASSFLAG Index) {
			return GetClassLongPtr(hWnd, Index);
		}
		/// ditto
		ULONG_PTR getClassInfo(int Index) {
			return GetClassLongPtr(hWnd, Index);
		}
		/// SetClassLongPtrラッパ。
		ULONG_PTR setClassInfo(CLASSFLAG Index, LONG_PTR NewInfo) {
			return SetClassLongPtr(hWnd, Index, NewInfo);
		}
		/// ditto
		ULONG_PTR setClassInfo(int Index, LONG_PTR NewInfo) {
			return SetClassLongPtr(hWnd, Index, NewInfo);
		}
	} else {
		/// GetClassLongラッパ。
		DWORD getClassInfo(CLASSFLAG Index) {
			return GetClassLong(hWnd, Index);
		}
		/// ditto
		DWORD getClassInfo(int Index) {
			return GetClassLong(hWnd, Index);
		}
		/// SetClassLongラッパ。
		ULONG_PTR setClassInfo(CLASSFLAG Index, LONG NewInfo) {
			return SetClassLong(hWnd, Index, NewInfo);
		}
		/// ditto
		ULONG_PTR setClassInfo(int Index, LONG NewInfo) {
			return SetClassLong(hWnd, Index, NewInfo);
		}
	}
	
	/// アイテムテキスト
	const Text text() {
		if(auto len = this.textLength()) {
			auto text=new wchar[len+1];
			
			if(GetWindowText(hWnd, text.ptr, len+1)) {
				return Text(text[0..len]);
			}
			
			throw new NeGuiException(Text("text"));
		}
		
		return Text.emptyText();
	}
	/// ditto
	bool text(in Text text) {
		return cast(bool)SetWindowText(hWnd, text.ptr);
	}
	bool number(T: int)(T Number) {
		try {
			return text(Text(Number));
		} catch(Throwable e) {
			return false;
		}
	}

	
	/// アイテムテキスト長取得
	const size_t textLength() {
		return GetWindowTextLength(hWnd);
	}
	
	/// 自身を終了させる。
	bool destroy() {
		return cast(bool)DestroyWindow(hWnd);
	}
	/// Windowsシステム上で有効か。
	const bool isAlive() {
		return cast(bool)IsWindow(hWnd);
	}
	deprecated alias isAlive alive;
	
	//// 有効無効
	bool enable(bool Enable) {
		return cast(bool)EnableWindow(hWnd, Enable);
	}
	/// ditto
	const bool enable() {
		return cast(bool)IsWindowEnabled(hWnd);
	}
	bool upDate() {
		return cast(bool)UpdateWindow(hWnd);
	}
	void reDraw(bool ReDraw) {
		send(WM_SETREDRAW, ReDraw, NONE);
	}
	/***/
	bool invaliDateRect(ref const(RECT) Rect, bool BackErase) {
		return cast(bool)InvalidateRect(hWnd, cast(RECT*)&Rect, BackErase);
	}
	/// ditto
	bool invaliDateRect(bool BackErase) {
		return cast(bool)InvalidateRect(hWnd, null, BackErase);
	}
	
	/**
	指定アイテムは子供か。

	Params:
		gui = 調べたい子供。

	Return:
		成功すればtrue、失敗すればfalseを返す。
	*/
	const bool isChild(in NeGui gui) {
		return cast(bool)IsChild(hWnd, gui.hWnd);
	}
	
	/**
	子アイテムの取得。

	Params:
		Id = 取得したいアイテムのID。

	Return:
		指定アイテムが有効ならそのアイテム。
	*/
	const NeGui child(ITEM_ID Id) {
		auto item=GetDlgItem(hWnd, Id);

		return item
			? NeGui.getGuiObject(item)
			: null
		;
	}
	/**
	クライアント座標から子アイテムの取得。

	Params:
		Point = クライアント座標。

	Return:
		子アイテム。
		指定座標に子アイテムが存在しなければ自身を返す。

	Exception:
		座標が変だった場合にはNeGuiExceptionを投げる。
	*/
	const NeGui child(ref const(POINT) Point) {
		auto Handle=ChildWindowFromPoint(hWnd, Point);

		enforce(Handle, new NeGuiException(Text("うん？")));

		return NeGui.getGuiObject(Handle);
	}
	/+
	private static {
		struct CHILDREN {
			NeGui[] list;
		}
		extern(Windows) BOOL EnumChildProc(HWND hWnd , LPARAM lParam) {
			if(auto gui=getGuiObject(hWnd)) {
				auto Children=cast(CHILDREN*)lParam;
				Children.list ~= gui;
			}
			return TRUE;
		}
	}
	const NeGui[] children() {
		CHILDREN Children;
		if(EnumChildWindows(hWnd, &EnumChildProc, cast(LPARAM)&Children)) {
			return Children.list;
		}
		throw new NeGuiException(ERR.toText);
	}
	+/
	typedef bool delegate(NeGui gui)EnumChildDelegate;
	private {
		struct ChildBox {
			EnumChildDelegate dg;
			size_t callIndex;
		}
		static extern(Windows) BOOL EnumChildProc(HWND hWnd , LPARAM lParam) {
			auto child=cast(ChildBox*)lParam;
			assert(child);
			child.callIndex++;
			if(auto ChildObject=getGuiObject(hWnd)) {
				return child.dg(ChildObject);
			}
			
			return true;
		}
	}
	const bool enumChild(EnumChildDelegate dg) {
		ChildBox child;
		child.dg = dg;
		if(EnumChildWindows(hWnd, &EnumChildProc, cast(LPARAM)&child)) {
			return true;
		}
		return false;
	}
	/**
	History:
		1.090:
			[P] EnumChildWindowsをdelegateに移行。
	*/
	const NeGui[] children() {
		NeGui[] list;

		auto dg = cast(EnumChildDelegate)(NeGui gui) {
			list ~= gui;
		};
		enumChild(dg);
		
		return list;
	}

	/**
	親取得。
	*/
	const NeGui parent() {
		if(auto hParent=GetParent(hWnd)) {
			if(auto gui=cast(NeGui)NeGui.getGuiObject(hParent)) {
				return gui;
			} else {
				return new OtherGui(hParent);
			}
		}
	
		return null;
	}
	const NeGui root() {
		if(auto gui=parent) {
			return gui.root;
		}
		
		return new OtherGui(hWnd);
	}

	static NeGui pointToGui(const ref POINT Point) {
		if(auto hGuiWnd=WindowFromPoint(cast(POINT)Point)) {
			if(auto gui=NeGui.getGuiObject(hGuiWnd)) {
				return gui;
			} else {
				return new OtherGui(hGuiWnd);
			}
		}

		return null;
	}

	///スクリーン座標→クライアント座標
	const bool pointToClient(ref POINT Point) {
		return cast(bool)ScreenToClient(hWnd, &Point);
	}
	///クライアント座標→スクリーン座標
	const bool pointToScreen(ref POINT Point) {
		return cast(bool)ClientToScreen(hWnd, &Point);
	}

	private enum GW_ENABLEDPOPUP = 6;
	/**
	History:
		1.00β16:
			[S] 名前をGWからRELATIONに変更。
	*/
	static enum RELATION {
		CHILD = GW_CHILD,     /// Z 順位が一番上の子ウィンドウを取得します。 
		FIRST = GW_HWNDFIRST, /// 指定したウィンドウと同じタイプのウィンドウで、Z 順位が一番上のウィンドウを取得します。 
		LAST  = GW_HWNDLAST,  /// 指定したウィンドウと同じタイプのウィンドウで、Z 順位が一番下のウィンドウを取得します。 
		NEXT  = GW_HWNDNEXT,  /// Z 順位が、指定したウィンドウの次にあるウィンドウを取得します。 
		PREV  = GW_HWNDPREV,  /// Z 順位が、指定したウィンドウの前にあるウィンドウを取得します。 
		OWNER = GW_OWNER,     /// 指定したウィンドウの親ウィンドウを取得します。
		ENABLEDPOPUP = GW_ENABLEDPOPUP,
	}
	const NeGui relation(RELATION Relation) {
		if(auto hRelationWnd=GetWindow(hWnd, Relation)) {
			if(auto negui=NeGui.getGuiObject(hRelationWnd)) {
				return negui;
			} else {
				return new OtherGui(hRelationWnd);
			}
		}

		return null;
	}
	
	/**
	ウィンドウをZ順位の一番手前に持ってきて、アクティブにします
	
	子ウィンドウの場合は、その子ウィンドウを所有する親ウィンドウがアクティブになります。

	Return:
		成功すればtrue、失敗すればfalseを返す。

	History:
		1.090:
			[S] 名前変更(topMost -> toTop)
	
	*/
	bool toTop() {
		return cast(bool)BringWindowToTop(hWnd);
	}
	///
	bool setShow(SHOW Show) {
		return cast(bool)ShowWindow(hWnd, Show);
	}
	///
	const bool isVisible() {
		return cast(bool)IsWindowVisible(hWnd);
	}
	/**
	フォアグラウンドへ設定。

	Return:
		成功すればtrue、失敗すればfalse。
	*/
	bool foreGround() {
		return cast(bool)SetForegroundWindow(hWnd);
	}
	

	///
	bool move(int x, int y, int Width, int Height, bool RePaint=true) {
		return cast(bool)MoveWindow(
			hWnd,
			x, y,
			Width, Height,
			RePaint
		);
	}
	/**
	History:
		1.00β17:
			新規作成。
	*/
	bool move(const ref RECT Rect, bool RePaint=true) {
		return move(Rect.left, Rect.top, Rect.right, Rect.bottom, RePaint);
	}
	///
	static enum AFTER {
		NONE      = 0,                       /// 無指定。
		BOTTOM    = cast(int)HWND_BOTTOM,    /// ウィンドウを Z 順位の一番下に設定します。最前面ウィンドウは同時に WS_EX_TOPMOST の指定が解除されます。 
		NOTOPMOST = cast(int)HWND_NOTOPMOST, /// 最前面ウィンドウ（WS_EX_TOPMOST）指定を解除します。 
		TOP       = cast(int)HWND_TOP,       /// ウィンドウを Z 順位の一番上に設定します。 
		TOPMOST   = cast(int)HWND_TOPMOST,   /// ウィンドウを最前面ウィンドウに設定します。最前面ウィンドウとは、常に最前面に表示されるウィンドウのことです。
	}
	///
	static enum SWP {
		DRAWFRAME      = SWP_DRAWFRAME, /// 再描画時に、ウィンドウの枠も再描画の対象とします。 
		FRAMECHANGED   = SWP_FRAMECHANGED, /// ウィンドウ サイズが変更されない場合でも、WM_NCCALCSIZE メッセージを送ります。 
		HIDEWINDOW     = SWP_HIDEWINDOW, /// ウィンドウを非表示にします。 
		NOACTIVATE     = SWP_NOACTIVATE, /// ウィンドウをアクティブ化しないようにします。 
		NOCOPYBITS     = SWP_NOCOPYBITS, /// ウィンドウ再配置時に、クライアント領域の内容を破棄します。 
		NOMOVE         = SWP_NOMOVE, /// ウィンドウの現在の位置を維持します。X パラメータ、Y パラメータは無視されます。 
		NOOWNERZORDER  = SWP_NOOWNERZORDER, /// オーナーウィンドウの Z 順位を変更しないようにします。
		NOREPOSITION   = SWP_NOREPOSITION, /// ditto
		NOREDRAW       = SWP_NOREDRAW, /// 変更結果を再描画しないようにします。 
		NOSENDCHANGING = SWP_NOSENDCHANGING, /// ウィンドウに WM_WINDOWPOSCHANGING メッセージを送らないようにします。 
		NOSIZE         = SWP_NOSIZE, /// ウィンドウの現在のサイズを維持します。cx パラメータ、cy パラメータは無視されます。 
		NOZORDER       = SWP_NOZORDER, /// 現在の Z 順位を維持します。hWndInsertAfter パラメータは無視されます。 
		SHOWWINDOW     = SWP_SHOWWINDOW, /// ウィンドウを表示します。 
	}
	///
	bool pos(AFTER After, int x, int y, int Width, int Height, SWP Flags) {
		return SetWindowPos(hWnd, cast(HWND)After, x, y, Width, Height, Flags)
			? true
			: false
		;
	}
	/**
	*/
	bool reLoad() {
		return this.pos(AFTER.NONE, 0, 0, 0, 0, SWP.DRAWFRAME | SWP.FRAMECHANGED| SWP.NOOWNERZORDER | SWP.NOSIZE | SWP.NOZORDER | SWP.NOMOVE);
	}
	/**
	History:
		1.090:
			[F] 最背面移動を強制。
	*/
	bool topMost(bool TopMpst) {
		return this.pos(TopMpst ? AFTER.TOPMOST: AFTER.BOTTOM, 0, 0, 0, 0, SWP.NOMOVE | SWP.NOSIZE | SWP.NOACTIVATE);
	}
	/**
	座標。
	*/
	bool position(in int x, in int y) {
		return this.pos(AFTER.NONE, x, y, 0, 0, SWP.DRAWFRAME | SWP.NOSIZE | SWP.NOZORDER);
	}
	/// ditto
	bool position(ref const(POINT) Point) {
		return this.position(Point.x, Point.y);
	}
	
	/**
	サイズ。
	*/
	bool size(in int Width, in int Height) {
		return this.pos(AFTER.NONE, 0, 0, Width, Height, SWP.DRAWFRAME | SWP.NOMOVE | SWP.NOZORDER);
	}
	/// ditto
	bool size(ref const(SIZE) Size) {
		return this.size(Size.cx, Size.cy);
	}

	/**
	アイテムのクライアントサイズを取得。

	Return:
		アイテムのクライアントサイズ。

	Exception:
		NeGuiException
	*/
	const ref SIZE clientSize() {
		RECT Rect=void;
		
		enforce(GetClientRect(hWnd, &Rect), new NeGuiException(ERR.toText));
		
		auto Size=new SIZE;
		Size.cx = Rect.right;
		Size.cy = Rect.bottom;

		return *Size;
	}

	/**
	アイテムのスクリーン座標を取得。

	Return:
		アイテムのスクリーン座標。

	Exception:
		NeGuiException
	*/
	const ref RECT itemRect() {
		auto Rect=new RECT;

		enforce(GetWindowRect(hWnd, Rect), new NeGuiException(ERR.toText));

		return *Rect;
	}

	/**
	アイテムのサイズを取得。

	Return:
		アイテムサイズ。

	Exception:
		NeGuiException
	*/
	const ref SIZE itemSize() {
		auto Rect=itemRect();

		auto Size=new SIZE;

		Size.cx = Rect.right  - Rect.left;
		Size.cy = Rect.bottom - Rect.top;

		return *Size;
	}

	const ref POINT itemPosition() {
		auto Rect=itemRect();
		auto Point=new POINT;

		Point.x = Rect.left;
		Point.y = Rect.top;

		return *Point;
	}

	/**
	呼び出し側スレッドのフォーカスアイテムを取得。

	Return:
		失敗時にはnull。

	History:
		1.00β14:
			[S] final?
	*/
	static NeGui getFocus() {
		auto Handle=GetFocus();

		return Handle
			? NeGui.getGuiObject(Handle)
			: null
		;
	}
	/**
	*/
	bool setFocus() {
		return cast(bool)SetFocus(hWnd);
	}

	/**
	*/
	bool lockUpDate(bool Lock) {
		return cast(bool)LockWindowUpdate(Lock ? hWnd: null);
	}

	/**
	デバイスコンテキストの取得。
	*/
	const HDC getDC() {
		return GetDC(hWnd);
	}
	/**
	デバイスコンテキストの解放。

	--------------------------------
	auto gui=new NeGui(HANDLE);

	auto hDC=gui.getDC();
	scope(exit) gui.releaseDC(hDC);
	--------------------------------
	*/
	bool releaseDC(HDC hDC) {
		return cast(bool)ReleaseDC(hWnd, hDC);
	}

	/// アイテムのCabvasを取得
	Canvas canvas() {
		return new Canvas(this);
	}
	/// WM_PAINTの糖衣
	Canvas beguinPaint(ref PAINTSTRUCT PaintStruct)
	out(r) {
		assert(!r.suicide);
	}
	body {
		return new Canvas(BeginPaint(hWnd, &PaintStruct), false);
	}
	/// ditto
	void endPaint(ref PAINTSTRUCT PaintStruct) {
		EndPaint(hWnd, &PaintStruct);
	}
	
	const Icon getIcon(Icon.FIXED IconFixed) {
		return new Icon(cast(HICON)send(WM_GETICON, IconFixed == Icon.FIXED.NORMAL ? ICON_BIG: ICON_SMALL, NONE), false);
	}
	Icon setIcon(Icon.FIXED IconFixed, in Icon icon) {
		HICON hIcon=cast(HICON)send(WM_SETICON, IconFixed == Icon.FIXED.NORMAL ? ICON_BIG: ICON_SMALL, icon ? cast(LPARAM)icon(): NONE);
		if(hIcon) {
			return new Icon(hIcon, false);
		}
		return null;
	}
	const Font font() {
		auto hFont=cast(HFONT)send(WM_GETFONT, NONE, NONE);

		return hFont
			? new Font(hFont, false)
			: null
		;
	}
	void font(in Font font) {
		send(WM_SETFONT, cast(WPARAM)font(), true);
	}

	///
	bool startHotKey(HOTKEY_ID Id, MOD Mod, KEY Key)
	in {
		version(gui) {
			assert(Id <= 0xBFFF);
		} else version(dll) {
			assert(0xC000 <= Id && Id <= 0xFFFF);
		}
	}
	body {
		return RegisterHotKey(hWnd, Id, Mod, Key)
			? true
			: false
		;
	}
	///
	bool stopHotKey(HOTKEY_ID Id) {
		return UnregisterHotKey(hWnd, Id)
			? true
			: false
		;
	}

	/**
	タイマー開始。
	*/
	bool startTimer(TIMER_ID Id, DWORD ms, TIMERPROC TimerProc=null)
	in {
		assert(Id > WM_APP);
	}
	body {
		return SetTimer(hWnd, Id, ms, TimerProc)
			? true
			: false
		;
	}
	
	bool stopTimer(TIMER_ID Id)
	in {
		assert(Id > WM_APP);
	}
	body {
		return KillTimer(hWnd, Id)
			? true
			: false
		;
	}

	void dropFile(bool Accept) {
		DragAcceptFiles(hWnd, Accept);
	}

}

template NeGuiDestructor() {
	~this() {
		if(hWnd) {
			if(auto me=getGuiObject(hWnd)) {
				if(me.isAlive) {
					me.destroy;
				}
				hWnd = null;
			}
		}
	}
}

class OtherGui: NeGui {
	this(HWND hWnd) {
		super(hWnd);
	}
}

NeGui GetGui(HWND hWnd) {
	if(hWnd) {
		if(auto gui=NeGui.getGuiObject(hWnd)) {
			return gui;
		} else {
			return new OtherGui(hWnd);
		}
	} else {
		return null;
	}
}

