﻿/**
ファイルとパス。

Phobosがchar[]大好きだったんで必要分だけ再構築。

History:
	1.000:
		[S] NeGuiが依存している部分あったんでpackageをNeGuiに移動、全然GUIじゃない。

	1.00β16:
		std.fileをimportしないように変更。
*/
module nemuxi.negui.file.file;

debug import std.stdio: wl = writefln, pl = printf;
debug(file) void main() {}

import std.contracts;
import std.conv;

import win32.windows;
import win32.shlwapi;

import nemuxi.negui.system.base;

unittest {
	assert(!(PATH.MAX_NAME % 4));
	assert(!(PATH.MAX_PATH % 4));
}

///
abstract class NeGuiFileBaseException: NeGuiException {
	mixin MixInNeGuiException;
}
///
class NeGuiPathException: NeGuiFileBaseException {
	mixin MixInNeGuiException;
}
///
class NeGuiFileException: NeGuiFileBaseException {
	mixin MixInNeGuiException;
}


static struct PATH {
	static:

	/**
	History:
		1.00β14:
			[S] nemuxi.baseから引越し。
	*/
	enum {
		MAX_NAME = 255 + 1,
		MAX_PATH = 32767 + MAX_NAME + 1,
	}

	/**
	フォルダパス判定。

	NeGui基準で最後が\をフォルダとする。

	Params:
		Path = 調べるパス。

	Return:
		フォルダならtrue、違えばfalse。
	*/
	bool isFolderPath(in Text Path) {
		return Path.length && Path[Path.length-1] == '\\';
	}
	debug(file) unittest {
		assert(isFolderPath(`asd\asd\`w.toText));
		assert(!isFolderPath(`asd\asd`c.toText));
	}
	
	/**
	パスをフォルダに。

	Params:
		Path = パス。

	Return:
		変換後のパス。
		Pathが元からフォルダならPath自身を返す。
	*/
	Text addFolderSep(in Text Path) {
		if(isFolderPath(Path)) {
			return cast(Text)Path;
		}
		return Path ~ '\\';
	}
	debug(file) unittest {
		assert(addFolderSep(Text(``)) == `\`);
		assert(addFolderSep(Text(`asd\asd\`)) == `asd\asd\`);
		assert(addFolderSep(Text(`asd\asd`)) == `asd\asd\`);
	}
	
	/**
	フォルダから非フォルダパスへ。

	Params:
		Path = パス。

	Return:
		変換後のパス。
	*/
	Text delFolderSep(in Text Path) {
		if(auto len=Path.length) {
			while(isFolderPath(Path[0..len])) {
				if(len--) continue; else break;
			}
			return Path[0..len];
		} else {
			return Text.emptyText();
		}
	}
	debug(file) unittest {
		assert(delFolderSep(Text(`asd\asd\`)) == `asd\asd`);
		assert(delFolderSep(Text(`asd\asd`)) == `asd\asd`);
		assert(delFolderSep(Text(``)) == ``);
		assert(delFolderSep(Text(`a`)) == `a`);
		assert(delFolderSep(Text(`a\\\`)) == `a`);
		assert(delFolderSep(Text(`LICENSE`)) == `LICENSE`);
	}
	/**
	パスの結合。

	Params:
		RootPath = 結合元となるパス。

		AddPath = RootPathに結合するパス。
	
	Return:
		結合後のパス。
	
	History:
		1.081:
			新規作成。
	*/
	Text join(in Text RootPath, in Text[] AddPath ... )
	out(r) {
		if(isFolderPath(AddPath[$-1])) {
			assert(isFolderPath(r));
		} else {
			assert(!isFolderPath(r));
		}
	}
	body {
		auto JoinPath=addFolderSep(RootPath);
		foreach(Path; AddPath) {
			JoinPath ~= addFolderSep(Path);
		}
		if(isFolderPath(AddPath[$-1])) {
			return JoinPath;
		} else {
			return delFolderSep(JoinPath);
		}
	}
	debug(file) unittest {
		Text t1, t2, t3;
		t1 = "a";
		t2 = "b";
		t3 = "c";
		assert(join(t1, t2, t3) == `a\b\c`);
		t3 ~= `\`;
		assert(join(t1, t2, t3) == `a\b\c\`);
	}

	/**
	拡張子取得。

	Params:
		Path = 拡張子を取得したいパス。

	Return:
		拡張子。
	*/
	Text getExtension(in Text Path) {
		auto p=PathFindExtension(Path.ptr);
		if(p) {
			return Text(cast(wchar*)(p+1));
		} else {
			return Text.emptyText;
		}
	}
	debug(file) unittest {
		assert(getExtension(Text(``)) == ``);
		assert(getExtension(Text(`123456`)) == ``);
		assert(getExtension(Text(`123.56`)) == `56`);
		assert(getExtension(Text(`.23.56`)) == `56`);
		assert(getExtension(Text(`.23456`)) == `23456`);
		assert(getExtension(Text(`abc\def`)) == Text(``));
		assert(getExtension(Text(`abc\.def`)) == Text(`def`));
		Text t;
		assert(getExtension(t) == Text(``));
	}
	
	/**
	拡張子の変更。

	Params:
		Path = パス名。

		Extension = 変更する拡張子。

	Return:
		Pathに拡張子が無い場合は.Extensionが追加される。

	History:
		1.063:
			新規作成。
	*/
	Text changeExtension(in Text Path, in Text Extension) {
		if(auto ExtLength=getExtension(Path).length) {
			return Path[0 .. Path.length - ExtLength] ~ Extension;
		} else {
			return Path ~ '.' ~ Extension;
		}
	}
	debug(file) unittest {
		assert(changeExtension(Text(``), Text(``)) == Text(`.`));
		assert(changeExtension(Text(`abc.def`), Text(`ghi`)) == Text(`abc.ghi`));
		assert(changeExtension(Text(`abc`), Text(`def`)) == Text(`abc.def`));
		assert(changeExtension(Text(`.abc`), Text(`def`)) == Text(`.def`));
	}

	
	/**
	ファイル名取得。

	Params:
		path = ファイル名を取得したいパス。

	Return:
		ファイル名。
		こいつは拡張子付き。
	*/
	Text getFileName(in Text path) {
		return Text(PathFindFileName(path.ptr));
	}
	debug(file) unittest {
		assert(getFileName(Text(``)) == ``);
		assert(getFileName(Text(`Z:\aaa.txt`)) == `aaa.txt`);
		assert(getFileName(Text(`Z:\aaatxt`)) == `aaatxt`);
		assert(getFileName(Text(`Z:\aaatxt\`)) == `aaatxt\`);
		assert(getFileName(Text(`LICENSE`)) == `LICENSE`);
	}
	/**
	ファイル名取得。

	Params:
		path = ファイル名を取得したいパス。

	Return:
		ファイル名。
		こいつは拡張子無し。

	History:
		1.00β16:
			[F] 拡張子のみ(.htaccess等の隠しファイル)をファイル名として扱うように変更。
	
		1.00β11:
			スライスで死ぬ対策。
	*/
	Text getName(in Text path) {
		auto name=delFolderSep(getFileName(path));
		
		auto ext =getExtension(path);
		
		if(ext.length && name.length > ext.length+1) {
			return name[0..name.length - ext.length-1];
		} else {
			return name;
		}
	}
	debug(file) unittest {
		assert(getName(Text(``)) == ``);
		assert(getName(Text(`.`)) == `.`);
		assert(getName(Text(`.aaa`)) == `.aaa`);
		assert(getName(Text(`Z:\aaa.txt`)) == `aaa`);
		assert(getName(Text(`Z:\aaatxt\`)) == `aaatxt`);
		assert(getName(Text(`aaabbb`)) == `aaabbb`);
		assert(getName(Text(`aaa.bbb`)) == `aaa`);
		assert(getName(Text(`aaa.bbb.ccc`)) == `aaa.bbb`);
	}
	/**
	親フォルダ取得。

	Params:
		親フォルダを取得したいパス。

	Return:
		親フォルダ。

	Out:
		戻り値に長さがあれば最後は\で終わってる。

	History:
		1.00β16:
			[P] Phobosを使用しない様に変更。
			[S] 名前をgetOwnerFolderからownerFolderに変更。
	*/
	Text ownerFolder(in Text path)
	out(r) {
		if(r.length) {
			assert(r[r.length-1] == '\\');
		}
	}
	body {
		if(path.length) {
			auto folder=delFolderSep(path);
			for(auto i=folder.length-1; i != 0; i--) {
				if(folder[i] == '\\') {
					return folder[0 .. i+1];
				} else if(folder[i] == ':') {
					return folder[0 .. i+1] ~ '\\';
				}
			}
		}

		return Text(`.\`);
	}
	debug(file) unittest {
		assert(ownerFolder(Text(``)) == `.\`);
		assert(ownerFolder(Text(`123`)) == `.\`);
		assert(ownerFolder(Text(`123\456`)) == `123\`);
		assert(ownerFolder(Text(`123\456\`)) == `123\`);
		assert(ownerFolder(Text(`123\456\789`)) == `123\456\`);
		assert(ownerFolder(Text(`C:\`)) == `C:\`);
	}

	/**
	ファイルパス比較。

	単純にファイル文字列の比較であって相対パス等は考慮しない。
	
	History:
		1.00β11:
			新規作成。
	*/
	int cmp(in Text path1, in Text path2) {
		return path1.cmpi(path2);
	}

	/**
	History:
		1.00β17:
			新規作成。
	*/
	Text modulePath(HINSTANCE hInstance, size_t Length=PATH.MAX_PATH) {
		auto text=new wchar[Length];

		if(!GetModuleFileName(hInstance, text.ptr, text.length)) {
			throw new NeGuiPathException(ERR.toText);
		}

		return Text(text.ptr);
	}

	/**
	History:
		1.00β17:
			[P] 取得をmodulePathに委譲。
	
		1.00β14:
			[P] メモリ確保用の引数追加。
			[S] バカっぽかったのを少しだけアホっぽく。
	
		1.00β11:
			nemuxi.baseから移動。
	*/
	private Text MyPath(size_t Length) {
		Text _MyPath() {
			try {
				return modulePath(GetModuleHandle(null), Length);
			} catch(NeGuiPathException e) {
				throw new NeGuiPathException(Text("自分の居場所さえ分からない"), e);
			}
		}
		// 最悪だ
		debug(nemuxi) {
			return _MyPath();
		} else debug {
			return Text(`Z:\nemuxi\nemuxi\nemuxi.exe`);
		} else {
			return _MyPath();
		}
	}
	/**
	自身のアドレス取得。
	
	History:
		1.00β14:
			MyPathの引数追加に合わせてデフォルト引数の追加。

		1.00β11:
			新規作成。
	*/
	Text myPath(size_t Length=PATH.MAX_PATH) {
		return MyPath(Length);
	}
	/**
	自身の親フォルダ。
	
	History:
		1.00β14:
			myPathの引数追加に合わせてデフォルト引数の追加。

		1.00β11:
			nemuxi.baseから移動。
	*/
	Text myFolder(size_t Length=PATH.MAX_PATH) {
		return ownerFolder(myPath(Length));
	}

	/**
	カレントディレクトリ取得。
	
	History:
		1.00β16:
			新規作成。
	*/
	Text currentFolder() {
		immutable BUFFER=GetCurrentDirectory(0, null);
		auto path=new wchar[BUFFER];

		auto CmpLength=GetCurrentDirectory(path.length, path.ptr);
		if(!CmpLength) {
			new NeGuiPathException(Text("カレントディレクトリ取得失敗"));
		}
		enforce(CmpLength == path.length-1, new NeGuiPathException(Text("カレントディレクトリ取得後に変わった？")));

		Text text;
		text.text = path[0 .. $];
		text.text[$-1] = '\\';
		
		return text;
	}
	///
	bool currentFolder(in Text Path, bool ThrowException=false) {
		auto ret=cast(bool)SetCurrentDirectory(Path.ptr);

		if(!ret) {
			enforce(ThrowException, new NeGuiPathException(ERR.toText));
		}
		
		return ret;
	}

}


static struct FILE {
	static:

	enum ATTRIBUTE {
		NONE                = 0xFFFFFFFF,
		ARCHIVE             = FILE_ATTRIBUTE_ARCHIVE,             /// アーカイブファイルまたはアーカイブディレクトリです。アプリケーションはこの属性を、ファイルのバックアップや削除のためのマークとして使います。
		COMPRESSED          = FILE_ATTRIBUTE_COMPRESSED,          /// 指定されたファイルまたはディレクトリは圧縮されています。ファイルの場合、ファイル内の全データが圧縮されていることを意味します。ディレクトリの場合、そのディレクトリ内に新しく作成されるファイルまたはサブディレクトリが、既定で圧縮状態になることを意味します。
		DEVICE              = FILE_ATTRIBUTE_DEVICE,              /// 予約済み。使わないでください。
		FOLDER              = FILE_ATTRIBUTE_DIRECTORY,           /// 指定されたハンドルは、ディレクトリに関連しています。
		ENCRYPTED           = FILE_ATTRIBUTE_ENCRYPTED,           /// 指定されたファイルまたはディレクトリは暗号化されています。ファイルの場合、ファイル内の全データストリームが暗号化されていることを意味します。ディレクトリの場合、そのディレクトリ内に新しく作成されるファイルまたはサブディレクトリが、既定で暗号化状態になることを意味します。
		HIDDEN              = FILE_ATTRIBUTE_HIDDEN,              /// 隠しファイルまたは隠しディレクトリです。通常のディレクトリリスティングでは表示されません。
		NORMAL              = FILE_ATTRIBUTE_NORMAL,              /// 指定されたファイルまたはディレクトリには、特に属性はありません。単独で返った場合にのみ、この属性は有効です。
		NOT_CONTENT_INDEXED = FILE_ATTRIBUTE_NOT_CONTENT_INDEXED, /// Windows 2000：このファイルは、「インデックスサービス」の対象になっていません。
		OFFLINE             = FILE_ATTRIBUTE_OFFLINE,             /// Windows 2000：このファイルのデータは、すぐには利用できません。この属性は、ファイルのデータがオフラインの記憶装置へ物理的に移動されたことを示します。この属性は、 Windows 2000 の階層記憶管理ソフトウェアである「リモート記憶域」が利用するものです。アプリケーションは、任意にこの属性を変更するべきではありません。
		READONLY            = FILE_ATTRIBUTE_READONLY,            /// このファイルまたはディレクトリは読み取り専用です。アプリケーションはこのファイルの読み取りを行えますが、書き込みや削除はできません。ディレクトリの場合、アプリケーションは削除を行えません。
		REPARSE_POINT       = FILE_ATTRIBUTE_REPARSE_POINT,       /// このファイルには、再解析ポイントが関連付けられています。
		SPARSE_FILE         = FILE_ATTRIBUTE_SPARSE_FILE,         /// このファイルは、スパースファイル（疎なファイル、未使用の領域が多い、または同じ値が長く続くファイル）です。
		SYSTEM              = FILE_ATTRIBUTE_SYSTEM,              /// このファイルまたはディレクトリは、オペレーティングシステムの一部、またはオペレーティングシステム専用です。
		TEMPORARY           = FILE_ATTRIBUTE_TEMPORARY,           /// このファイルは、一時ファイルとして使われています。ファイルシステムは、データをハードディスクのような大容量記憶装置へ書き込む代わりに、高速なアクセスが行えるよう、すべてのデータをメモリ内に維持することを試みます。アプリケーションは、必要がなくなった段階で一時ファイルをすぐに削除するべきです。
	}
	ATTRIBUTE attributes(in Text path) {
		return cast(ATTRIBUTE)GetFileAttributes(path.ptr);
	}
	bool attributes(in Text path, ATTRIBUTE Attribute) {
		return cast(bool)SetFileAttributes(path.ptr, Attribute);
	}
	/**
	指定アドレスは存在するか。

	Params:
		path = 調べたいアドレス。

	Return:
		存在すればtrue、しなければfalse。
	*/
	bool isExistence(in Text path) {
		return attributes(path) != ATTRIBUTE.NONE;
	}
	/**
	指定アドレスはフォルダか。

	Params:
		path = 調べたいアドレス。

	Return:
		フォルダならtrue。

	Exception:
		アドレス自体が存在しなければNeGuiFileException。
	*/
	bool isFolder(in Text path) {
		enforce(isExistence(path), new NeGuiFileException(Text("そもそもファイル無い。")));
		return (attributes(path) & ATTRIBUTE.FOLDER) != 0;
	}
	/**
	指定アドレスはファイルか。

	Params:
		path = 調べたいアドレス。

	Return:
		ファイルならtrue。

	Exception:
		アドレス自体が存在しなければNeGuiFileException。
	*/
	bool isFile(in Text path) {
		return !isFolder(path);
	}

	
	/**
	フォルダ作成。

	Params:
		path = フォルダ
	*/
	bool makeFolder(in Text path, bool Recurse=true) {
		if(Recurse) {
			const left=PATH.ownerFolder(path);
			isExistence(left) || makeFolder(left, true);
			return makeFolder(path, false);
		} else {
			return cast(bool)CreateDirectory(PATH.delFolderSep(path).ptr, null);
		}
	}
	
	bool deleteFile(in Text path) {
		return cast(bool)DeleteFile(path.ptr);
	}
	bool deleteFolder(in Text path) {
		return cast(bool)RemoveDirectory(path.ptr);
	}

	bool move(in Text SrcPath, in Text NewPath) {
		return cast(bool)MoveFile(SrcPath.ptr, NewPath.ptr);
	}
	bool copy(in Text SrcPath, in Text NewPath, bool Overwrite) {
		return cast(bool)CopyFile(SrcPath.ptr, NewPath.ptr, !Overwrite);
	}
	
	enum DRIVE {
		UNKNOWN     = DRIVE_UNKNOWN,     /// 不明です。 
		NO_ROOT_DIR = DRIVE_NO_ROOT_DIR, /// 指定したルートディレクトリは存在しません。 
		REMOVABLE   = DRIVE_REMOVABLE,   /// ドライブからディスクを抜くことができます。 
		FIXED       = DRIVE_FIXED,       /// ドライブからディスクを抜くことができません。 
		REMOTE      = DRIVE_REMOTE,      /// ネットワーク ドライブです。 
		CDROM       = DRIVE_CDROM,       /// CD-ROM ドライブです。 
		RAMDISK     = DRIVE_RAMDISK,     /// RAM ディスク ドライブです。 
	}
	DRIVE driveType(in Text Drive) {
		return cast(DRIVE)GetDriveType(Drive.ptr);
	}

	
	



}

deprecated Text GetNameEx(in Text path) {
	Text Name;
	/+
	if(IsPath(path)) {
		
	} else {
	}
	+/
	
	if(path.length) {
		Name = PATH.ownerFolder(path);
	}
	if(!Name.length) {
		Name = PATH.ownerFolder(path[0..path.length-1]);
		if(!Name.length) {
			if(FILE.isExistence(path)) {
				// ドライブ？
				Name = path;
			} else {
				Name = path;
			}
		}
	}
	
	return Name;
}





