﻿/**
メモリ情報。

*/
module nemuxi.negui.environ.memory;

debug import std.stdio: wl = writefln, pl = printf;
debug(memory) void main() {}

import std.contracts;

import win32.windows;

import nemuxi.negui.system.base;
import nemuxi.negui.environ.environ;

///
class MemoryException: EnvironException {
	mixin MixInNeGuiException;
}

class Memory: IInitialize {
	version(X86_64) {
		// 多分…
		alias size_t SIZE_T;
	} else {
		alias ulong SIZE_T;
	}
	
	private {
		SYSTEM_INFO    SystemInfo;
		MEMORYSTATUS   MemoryStatus;
		MEMORYSTATUSEX MemoryStatusEx;
	}

	override void initialize() {
		GetSystemInfo(&SystemInfo);
		
		MemoryStatus.dwLength = MemoryStatus.sizeof;
		GlobalMemoryStatus(&MemoryStatus);
		MemoryStatusEx.dwLength = MemoryStatusEx.sizeof;
		enforce(GlobalMemoryStatusEx(&MemoryStatusEx), new MemoryException(ERR.toText));
	}

	mixin(SMixInClassGet!(size_t)("pageSize", q{SystemInfo.dwPageSize}, false));
	
	mixin(SMixInClassGet!(ptrdiff_t)("minAddress", q{SystemInfo.lpMinimumApplicationAddress}, false));
	mixin(SMixInClassGet!(ptrdiff_t)("maxAddress", q{SystemInfo.lpMaximumApplicationAddress}, false));
	
	mixin(SMixInClassGet!(size_t)("minAllocation", q{SystemInfo.dwAllocationGranularity}, false));

	mixin(SMixInClassGet!(SIZE_T)("ramTotalSize", q{MemoryStatusEx.ullTotalPhys}, false));
	mixin(SMixInClassGet!(SIZE_T)("ramFreeSize",  q{MemoryStatusEx.ullAvailPhys}, false));
	mixin(SMixInClassGet!(SIZE_T)("ramUsingSize", q{ramTotalSize - ramFreeSize}, false));

	mixin(SMixInClassGet!(SIZE_T)("pageTotalSize", q{MemoryStatusEx.ullTotalPageFile}, false));
	mixin(SMixInClassGet!(SIZE_T)("pageFreeSize",  q{MemoryStatusEx.ullAvailPageFile}, false));
	mixin(SMixInClassGet!(SIZE_T)("pageUsingSize", q{pageTotalSize - pageFreeSize}, false));

	mixin(SMixInClassGet!(SIZE_T)("virtualTotalSize", q{MemoryStatusEx.ullTotalVirtual}, false));
	mixin(SMixInClassGet!(SIZE_T)("virtualFreeSize",  q{MemoryStatusEx.ullAvailVirtual}, false));
	mixin(SMixInClassGet!(SIZE_T)("virtualUsingSize", q{virtualTotalSize - virtualFreeSize}, false));
	mixin(SMixInClassGet!(SIZE_T)("virtualAvailExtended", q{MemoryStatusEx.ullAvailExtendedVirtual}, false)); // 仮想メモリ拡張空き容量

	const Text toText() {
		return Text(
			"%s: %08X / %08X", "Address,min-max",  minAddress, maxAddress,
			Text.newline,
			InfoTextLine("MinAllocation", minAllocation),
			Text.newline,
			InfoTextLine("PageSize",      pageSize),
			Text.newline,
			"%s: %s / %s", "Page,use/all",  pageFreeSize, pageTotalSize,
			Text.newline,
			"%s: %s / %s", "Ram,use/all",  ramUsingSize,  ramTotalSize,
			Text.newline,
			"%s: %s / %s(%s)", "Virtual,use/all(ext)",  virtualUsingSize, virtualTotalSize, virtualAvailExtended
		);
	}
	override string toString() {
		return toText.text8;
	}
}
debug(memory) unittest {
	auto m=new Memory;
	m.initialize;
	wl("%s", m);
}


