﻿/**
あ
*/
module nemuxi.negui.control.treeview.treeview;

debug import std.stdio: wl = writefln, pl = printf;
debug(treeview) void main() {}

import std.bitmanip;
import std.contracts;

import win32.windows;
import win32.commctrl;

import nemuxi.negui.system.base;
import nemuxi.negui.draw.color;
import nemuxi.negui.draw.imagelist;
import nemuxi.negui.system.type.basic;
import nemuxi.negui.system.type.enumerated;
import nemuxi.negui.system.type.record;
public import nemuxi.negui.negui;
public import nemuxi.negui.control.control;

///
class TreeViewException: ControlException {
	mixin MixInNeGuiException;
}

/**
History:
	1.010:
		[S] ツリーアイテムの型を統一。
*/
typedef HTREEITEM TREE_NODE=TVI_ROOT;

enum {
	TVIS_EXPANDPARTIAL=0x0080,
	TVSI_NOSINGLEEXPAND=0x8000,
}

enum {
  TVN_ASYNCDRAW           = TVN_FIRST-20,
  TVN_ITEMCHANGINGA       = TVN_FIRST-16,
  TVN_ITEMCHANGINGW       = TVN_FIRST-17,
  TVN_ITEMCHANGEDA        = TVN_FIRST-18,
  TVN_ITEMCHANGEDW        = TVN_FIRST-19,
}
version(UNICODE) enum {
  TVN_ITEMCHANGING        = TVN_ITEMCHANGINGW,
  TVN_ITEMCHANGED         = TVN_ITEMCHANGEDW,
} else enum {
  TVN_ITEMCHANGING        = TVN_ITEMCHANGINGA,
  TVN_ITEMCHANGED         = TVN_ITEMCHANGEDA,
}



class TreeView: Control, ICommonControl, INoHideSelect {
	alias BUFFER.TEXT TEXTLENGTH;

	enum EVENT: MESSAGETYPE {
		ASYNCDRAW      = TVN_ASYNCDRAW     ,
		BEGINDRAG      = TVN_BEGINDRAG     ,
		BEGINLABELEDIT = TVN_BEGINLABELEDIT,
		BEGINRDRAG     = TVN_BEGINRDRAG    ,
		DELETEITEM     = TVN_DELETEITEM    ,
		ENDLABELEDIT   = TVN_ENDLABELEDIT  ,
		GETDISPINFO    = TVN_GETDISPINFO   ,
		GETINFOTIP     = TVN_GETINFOTIP    ,
		ITEMCHANGED    = TVN_ITEMCHANGED   ,
		ITEMCHANGING   = TVN_ITEMCHANGING  ,
		ITEMEXPANDED   = TVN_ITEMEXPANDED  ,
		ITEMEXPANDING  = TVN_ITEMEXPANDING ,
		KEYDOWN        = TVN_KEYDOWN       ,
		SELCHANGED     = TVN_SELCHANGED    ,
		SELCHANGING    = TVN_SELCHANGING   ,
		SETDISPINFO    = TVN_SETDISPINFO   ,
		SINGLEEXPAND   = TVN_SINGLEEXPAND  ,
	}
	mixin MixInCommonControl;

	/**
	History:
		1.032:
			[P] 処理内容変更。
	*/
	this(NeGui Owner, ITEM_ID Id) {
		NEGUIINFO NeGuiInfo;
		
		NeGuiInfo.style |= TVS_HASBUTTONS | TVS_HASLINES | TVS_LINESATROOT;
		NeGuiInfo.exStyle |= WS_EX_CLIENTEDGE;

		NeGuiInfo.owner     = Owner;
		NeGuiInfo.id        = Id;
		
		this(NeGuiInfo);

		noHideSelect = true;
	}

	/**
	History:
		1.032:
			新規作成。
	*/
	override this(ref NEGUIINFO NeGuiInfo) {
		NeGuiInfo.className = WC_TREEVIEW;

		super(NeGuiInfo);
	}

	/**
	指定アイテムの削除。

	Params:
		Item = 削除するアイテム

	Return:
		成功すればtrue、失敗すればfalse。
	*/
	bool del(TREE_NODE Node) {
		return super.send(TVM_DELETEITEM, NONE, cast(LPARAM)Node)
			? true
			: false
		;
	}
	
	static enum EXPAND {
		COLLAPSE       = TVE_COLLAPSE, /// 子アイテムのリストを閉じます。
		EXPAND         = TVE_EXPAND,   /// 子アイテムのリストを開きます。
		TOGGLE         = TVE_TOGGLE,   /// 子アイテムのリストが開かれている場合にはリストを閉じ、閉じられている場合にはリストを開きます。
		EXPAND_CHILD   = TVE_EXPAND | TVE_EXPANDPARTIAL,   /// Version 4.70 以上： 子アイテムのリストの一部のみを開きます。
		COLLAPSE_CHILD = TVE_COLLAPSE | TVE_COLLAPSERESET, /// 子アイテムのリストを閉じて、子アイテムを削除します。
	}
	bool expand(TREE_NODE Node, EXPAND Expand) {
		return cast(bool)send(TVM_EXPAND, cast(WPARAM)Expand, cast(LPARAM)Node);
	}
	
	/**
	背景色の取得。

	Return:
		取得した背景色。

	Exception:
		背景色がシステムカラーならTreeViewException。
	*/
	COLOR backColor() {
		uint code=send(TVM_GETBKCOLOR, NONE, NONE);
		
		if(code != -1) {
			return COLOR(code);
		}
		
		throw new TreeViewException(Text("背景色はシステムカラー"));
	}
	/**
	背景色の設定。

	Params:
		cl = 色。
		     nullの場合はシステムカラー。
	*/
	void backColor(in COLOR* cl) {
		uint code;
		if(cl) {
			code = cl.code;
		} else {
			code = -1;
		}
		send(TVM_SETBKCOLOR, NONE, code);
	}
	///
	bool isSystemBackColor() {
		return send(TVM_GETBKCOLOR, NONE, NONE) == -1;
	}

	int count() {
		return send(TVM_GETCOUNT, NONE, NONE);
	}
	int countVisible() {
		return send(TVM_GETVISIBLECOUNT, NONE, NONE);
	}

	bool get(ref TREEITEM TreeItem) {
		return send(TVM_GETITEM, NONE, cast(LPARAM)TreeItem.ptr)
			? true
			: false
		;
	}
	ref TREEITEM get(TREE_NODE TreeHandle) {
		auto TreeItem=new TREEITEM;
		auto TextBuf=new wchar[TEXTLENGTH];

		with(TREEITEM.MASK) TreeItem.mask = TEXT | IMAGE | DATA | STATE | HANDLE | SELECTEDIMAGE | CHILDREN | INTEGRAL | DI_SETITEM;
		with(TREEITEM.STATE)TreeItem.stateMask = SELECTED | CUT | DROPHILITED | BOLD | EXPANDED | EXPANDEDONCE | EXPANDPARTIAL | FOCUSED | OVERLAYMASK | STATEIMAGEMASK | USERMASK;

		TreeItem.node = TreeHandle;
		TreeItem.text = TextBuf.ptr;
		TreeItem.textLength = TEXTLENGTH;

		get(*TreeItem);

		return *TreeItem;
	}
	TREE_NODE insert(const ref TREEINSERTITEM InsertItem) {
		return cast(TREE_NODE)send(TVM_INSERTITEM, NONE, cast(LPARAM)&InsertItem.TvInsertItem);
	}
	bool set(ref const(TREEITEM) TreeItem)
	in {
		assert(TreeItem.node);
	}
	body {
		return send(TVM_SETITEM, NONE, cast(LPARAM)&TreeItem.TvItem)
			? true
			: false
		;
	}
	

	///
	static enum NEXT {
		ROOT            = TVGN_ROOT           , /// ツリービューのルート（最も上の階層）のアイテムを取得します。
		NEXT            = TVGN_NEXT           , /// 指定されたアイテムの同じグループ内の次のアイテムを取得します。
		PREVIOUS        = TVGN_PREVIOUS       , /// 指定されたアイテムの同じグループ内の前のアイテムを取得します。
		PARENT          = TVGN_PARENT         , /// 指定されたアイテムの親アイテムを取得します。
		CHILD           = TVGN_CHILD          , /// 指定されたアイテムが持つ最初の子アイテムを取得します。
		FIRSTVISIBLE    = TVGN_FIRSTVISIBLE   , /// ツリービューウィンドウ内で見えている最初のアイテムを取得します。
		NEXTVISIBLE     = TVGN_NEXTVISIBLE    , /// 指定されたアイテムに続く次の見えているアイテムを取得します。指定されたアイテムはツリービューウィンドウ内で見えていなければなりません。
		PREVIOUSVISIBLE = TVGN_PREVIOUSVISIBLE, /// 指定されたアイテムの前にある見えているアイテムを取得します。指定されたアイテムはツリービューウィンドウ内で見えていなければなりません。
		DROPHILITE      = TVGN_DROPHILITE     , /// ドラッグ・アンド・ドロップのターゲットとなっているアイテムを取得します。
		CARET           = TVGN_CARET          , /// 現在選択されているアイテムを取得します。
		LASTVISIBLE     = TVGN_LASTVISIBLE    , /// Version 4.71 以上：最後に広げられたアイテムを取得します。
	}
	///
	TREE_NODE getNode(TREE_NODE Item, NEXT Next) {
		return cast(TREE_NODE)send(TVM_GETNEXTITEM, cast(WPARAM)Next, cast(LPARAM)Item);
	}
	///
	TREE_NODE nodeRoot(TREE_NODE Item) {
		return getNode(Item, NEXT.ROOT);
	}
	///
	TREE_NODE nodeParent(TREE_NODE Item) {
		return getNode(Item, NEXT.PARENT);
	}
	///
	TREE_NODE nodeNext(TREE_NODE Item) {
		return getNode(Item, NEXT.NEXT);
	}
	///
	TREE_NODE nodePrevious(TREE_NODE Item) {
		return getNode(Item, NEXT.PREVIOUS);
	}
	///
	TREE_NODE nodeChild(TREE_NODE Item) {
		return getNode(Item, NEXT.CHILD);
	}
	///
	TREE_NODE nodeCaret() {
		return getNode(TREEINSERTITEM.ROOT, NEXT.CARET);
	}
	///
	TREE_NODE nodeHead(TREE_NODE BaseNode) {
		return nodeChild(nodeParent(BaseNode));
	}
	///
	TREE_NODE nodeTail(TREE_NODE BaseNode) {
		TREE_NODE TailNode;
		auto TempNode = BaseNode;
		while(cast(bool)(TempNode = nodeNext(TempNode))) {
			TailNode = TempNode;
		}
		return TailNode;
	}
	/**
	History:
		1.010:
			新規作成。
	*/
	TREE_NODE[] nodeSibling(TREE_NODE BaseNode, size_t InitialValue=BUFFER.INITIAL, size_t Increment=BUFFER.INCREMENT)
	in {
		assert(InitialValue);
		assert(Increment);
	}
	body {
		if(auto Node=nodeHead(BaseNode)) {
			size_t i=0;
			auto TreeNodes=new TREE_NODE[InitialValue];
			
			while(Node) {
				TreeNodes.autoIncrement(i, Increment);
				TreeNodes[i++] = Node;
				Node = nodeNext(Node);
			}

			return TreeNodes[0..i];
		}

		return null;
	}

	/**
	History:
		1.010:
			新規作成。
	*/
	TREEITEM[] getSibling(TREE_NODE BaseNode, size_t InitialValue=BUFFER.INITIAL, size_t Increment=BUFFER.INCREMENT)
	in {
		assert(InitialValue);
		assert(Increment);
	}
	body {
		if(auto Siblings=nodeSibling(BaseNode, InitialValue, Increment)) {
			auto TreeItems=new TREEITEM[Siblings.length];
			foreach(i, ref TreeItem; TreeItems) {
				TreeItem = get(Siblings[i]);
			}
			return TreeItems;
		}

		return null;
	}

	COLOR textColor() {
		uint code=send(TVM_GETTEXTCOLOR, NONE, NONE);
		
		if(code != -1) {
			return COLOR(code);
		}
		
		throw new TreeViewException(Text("前景色はシステムカラー"));
	}
	///
	bool isSystemTextColor() {
		return send(TVM_GETTEXTCOLOR, NONE, NONE) == -1;
	}
	void textColor(COLOR* cl) {
		uint code;
		if(cl) {
			code = cl.code;
		} else {
			code = -1;
		}
		send(TVM_SETTEXTCOLOR, NONE, code);
	}

	private static enum IMAGE_TYPE {
		NORMAL = TVSIL_NORMAL,
		STATE  = TVSIL_STATE,
	}
	private ImageList GetSetImageList(bool GetList, ImageList NewImage, IMAGE_TYPE ImageType) {
		auto hImageList = cast(HIMAGELIST)(GetList
			? send(TVM_GETIMAGELIST, ImageType, NONE)
			: send(TVM_SETIMAGELIST, ImageType, cast(LPARAM)NewImage())
		);

		return hImageList
			? new ImageList(hImageList, false)
			: null
		;
	}
	/***/
	ImageList imageListNormal() {
		return GetSetImageList(true, null, IMAGE_TYPE.NORMAL);
	}
	/// ditto
	ImageList imageListNormal(ImageList NewImage) {
		return GetSetImageList(false, NewImage, IMAGE_TYPE.NORMAL);
	}
	/***/
	ImageList imageListState() {
		return GetSetImageList(true, null, IMAGE_TYPE.STATE);
	}
	/// ditto
	ImageList imageListState(ImageList NewImage) {
		return GetSetImageList(false, NewImage, IMAGE_TYPE.STATE);
	}

	bool sort(TREE_NODE Item, bool Recurse) {
		return send(TVM_SORTCHILDREN, cast(WPARAM)Recurse, cast(LPARAM)Item)
			? true
			: false
		;
	}
	
	mixin(SMixInItemStyleGetSet("itemDrag", q{TVS_DISABLEDRAGDROP}));
	mixin(SMixInItemStyleGetSet("editLabel", q{TVS_EDITLABELS}));
	mixin(SMixInItemStyleGetSet("button", q{TVS_HASBUTTONS}));
	mixin(SMixInItemStyleGetSet("line", q{TVS_HASLINES}));
	mixin(SMixInItemStyleGetSet("rootLine", q{TVS_LINESATROOT}));
	mixin(SMixInItemStyleGetSet("noHideSelect", q{TVS_SHOWSELALWAYS}, true));
	mixin(SMixInItemStyleGetSet("checkBox", q{TVS_CHECKBOXES}));
	

	///
	TREE_NODE hitTest(ref TREEHITTEST TreeHitTese) {
		return cast(TREE_NODE)send(TVM_HITTEST, NONE, cast(LPARAM)TreeHitTese.ptr);
	}

	/**
	ノードコピー。
	ABからコピー。

	Params:
		Src = コピー元。

		Parent = コピー先親アイテム。

		InsertAfter = Parentに対する位置。
	*/
	TREE_NODE nodeCopy(in TREE_NODE SrcItem, in TREE_NODE Parent, in TREE_NODE InsertAfter) {
		TREEINSERTITEM TreeInsertItem;
		wchar[TEXTLENGTH] TextBuf;

		with(TREEITEM.MASK) TreeInsertItem.item.mask = TEXT | IMAGE | DATA | STATE | HANDLE | SELECTEDIMAGE | CHILDREN | INTEGRAL | DI_SETITEM;
		with(TREEITEM.STATE)TreeInsertItem.item.stateMask = DROPHILITED | EXPANDED | EXPANDEDONCE | EXPANDPARTIAL;

		TreeInsertItem.item.node = SrcItem;
		TreeInsertItem.item.text = TextBuf.ptr;
		TreeInsertItem.item.textLength = TEXTLENGTH;

		enforce(get(*TreeInsertItem.item), new TreeViewException(ERR.toText));

		TreeInsertItem.parent = Parent;
		TreeInsertItem.insertAfter = InsertAfter;

		immutable TempItem = insert(TreeInsertItem);
		auto ChildItem = getNode(SrcItem, NEXT.CHILD);

		while(ChildItem) {
			ChildItem = nodeCopy(ChildItem, TempItem, TREEINSERTITEM.LAST);
			ChildItem = getNode(ChildItem, NEXT.NEXT);
		}
		
		return SrcItem;
	}

	/**
	ノード移動。

	親を範囲として(視覚上の)上下に移動。

	Params:
		TargetItem = 移動させるノード。
		             正常終了後は死ぬ。

		Count = 上の場合は負数、下の場合は正数を指定。

	Return:
		移動後のノード。

	In:
		Count != 0
	*/
	TREE_NODE nodeMove(in TREE_NODE TargetItem, int Count)
	in {
		assert(Count);
	}
	body {
		immutable Parent = nodeParent(TargetItem);
		immutable MOVE_UP = Count > 0;
		bool Last=false;
		TREE_NODE InsertAfter;
		TREE_NODE SelecteItem;

		if(MOVE_UP) {
			if(auto TempAfter=nodePrevious(nodePrevious(TargetItem))) {
				InsertAfter = TempAfter;
			} else {
				InsertAfter = TREEINSERTITEM.FIRST;
				Last=true;
			}
		} else {
			if(auto TempAfter=nodeNext(TargetItem)) {
				InsertAfter = TempAfter;
			} else {
				InsertAfter = TREEINSERTITEM.LAST;
				Last=true;
			}
		}
		
		nodeCopy(TargetItem, Parent, InsertAfter);
		
		if(MOVE_UP) {
			if(Last) {
				SelecteItem = nodeHead(TargetItem);
			} else {
				SelecteItem = nodeNext(InsertAfter);
			}
		} else {
			if(Last) {
				SelecteItem = nodeTail(TargetItem);
			} else {
				SelecteItem = nodeNext(InsertAfter);
			}
		}
		
		del(TargetItem);
		
		select(SelecteItem, TreeView.SELECT.CARET);

		return SelecteItem;
	}

	
	///
	enum SELECT {
		NONE=0,
		CARET          = TVGN_CARET,
		DROPHILITE     = TVGN_DROPHILITE,
		FIRSTVISIBLE   = TVGN_FIRSTVISIBLE,
		NOSINGLEEXPAND = TVSI_NOSINGLEEXPAND,
	}
	///
	bool select(TREE_NODE TreeHandle, SELECT Select) {
		return cast(bool)send(TVM_SELECTITEM, Select, cast(WPARAM)TreeHandle);
	}

}
unittest {
	assert(TREEITEM.sizeof == TVITEMEX.sizeof, "設計が狂う");
}
///
struct TREEITEM {
	TVITEMEX TvItem;
	mixin(SMixInStructHiddenOriginal!(TVITEMEX)(`TvItem`));

	static enum MASK {
		TEXT          = TVIF_TEXT         , /// pszText, cchTextMax
		IMAGE         = TVIF_IMAGE        , /// iImage
		DATA          = TVIF_PARAM        , /// lParam
		STATE         = TVIF_STATE        , /// state, stateMask
		NODE          = TVIF_HANDLE       , /// hItem
		SELECTEDIMAGE = TVIF_SELECTEDIMAGE, /// iSelectedImage
		CHILDREN      = TVIF_CHILDREN     , /// cChildren
		INTEGRAL      = TVIF_INTEGRAL     , /// iIntegral
		DI_SETITEM    = TVIF_DI_SETITEM   , /// ツリービューコントロールが与えられた情報を保持し、再度情報を要求しないようにします。このフラグは TVN_GETDISPINFO 通知メッセージを処理する場合にのみ有効です。

		PARAM            = DATA,
		USERDATA         = PARAM        , /// lParam
		HANDLE           = NODE         , /// hItem
	}
	mixin(SMixInStructGetSet!(MASK)("mask", q{TvItem.mask}));

	mixin(SMixInStructGetSet!(TREE_NODE)("node", q{TvItem.hItem}));
	deprecated alias node item;

	static enum STATE {
		SELECTED       = TVIS_SELECTED       , /// アイテムは選択されています。
		CUT            = TVIS_CUT            , /// アイテムはカット・アンド・ペーストの操作の一部として選択されています。
		DROPHILITED    = TVIS_DROPHILITED    , /// アイテムはドラッグ・アンド・ドロップのターゲットとして選択されています。
		BOLD           = TVIS_BOLD           , /// アイテムは太字表示されています。
		EXPANDED       = TVIS_EXPANDED       , /// 子アイテムのリストが展開されています。このフラグは子アイテムを持つ親アイテムのみに適用されます。
		EXPANDEDONCE   = TVIS_EXPANDEDONCE   , /// 少なくとも一度以上、アイテムが持つ子アイテムのリストが展開されたことがあることを示します。このフラグは子アイテムを持つ親アイテムのみに適用されます。 TVM_EXPAND メッセージによってこのフラグがセットされた親アイテムに対して、 TVN_ITEMEXPANDING や TVN_ITEMEXPANDED 通知メッセージは発生しません。 メッセージで、 TVE_COLLAPSE と TVE_COLLAPSERESET フラグを指定したときにこのフラグはリセットされます。
		EXPANDPARTIAL  = TVIS_EXPANDPARTIAL  , /// Version 4.70 以上： アイテムが部分的に展開されています。
		FOCUSED        = TVIS_FOCUSED        , /// 項目にフォーカスがある
		OVERLAYMASK    = TVIS_OVERLAYMASK    , /// 項目のオーバーレイイメージが描画時に含められる
		STATEIMAGEMASK = TVIS_STATEIMAGEMASK , /// 項目の状態イメージが描画時に含められる
		USERMASK       = TVIS_USERMASK       , /// TVISSTATEIMAGEMAS と同じ
	}


	/// あ～もう知らん
	union STATUS {
		UINT Status;
		
		mixin(bitfields!(
			STATE, "State",       8,
			int,   "NormalIndex", 4,
			int,   "StateIndex",  4
		));

		static STATUS opCall(UINT StatusArg) {
			STATUS Status;
			Status.Status = StatusArg;
			return Status;
		}
	}

	STATUS status() {
		return STATUS(TvItem.state);
	}
	void status(STATUS Status) {
		TvItem.state = Status.Status;
	}
	
	mixin(SMixInStructGetSet!(STATE)("stateMask", q{TvItem.stateMask}));
	///
	const Text text() {
		return Text(cast(wchar*)TvItem.pszText);
	}
	///
	void text(wchar* s) {
		TvItem.pszText = s;
	}
	///
	wchar* text(in Text t) {
		return TvItem.pszText = t.ptr;
	}
	bool textCallBack() {
		return TvItem.pszText == LPSTR_TEXTCALLBACK;
	}
	void textCallBack(bool CallBack) {
		TvItem.pszText = CallBack ? LPSTR_TEXTCALLBACK: null;
	}
	mixin(SMixInStructGetSet!(int)("textLength", q{TvItem.cchTextMax}));

	mixin(SMixInStructGetSetCallBack!(int)("normalImageIndex", "normalImageCallBack", q{TvItem.iImage}, q{I_IMAGECALLBACK}));

	mixin(SMixInStructGetSetCallBack!(int)("selectedImageIndex", "selectedImageCallBack", q{TvItem.iSelectedImage}, q{I_IMAGECALLBACK}));
	
	mixin(SMixInStructGetSetCallBack!(int)("children", "childrenCallBack", q{TvItem.cChildren}, q{I_CHILDRENCALLBACK}));

	mixin(SMixInStructGetSet!(void*)("data", q{TvItem.lParam}));
	mixin(SMixInStructGetSet!(void*)("integral", q{TvItem.iIntegral}));
}

///
struct TREEINSERTITEM {
	TVINSERTSTRUCT TvInsertItem;
	mixin(SMixInStructHiddenOriginal!(TVINSERTSTRUCT)(`TvInsertItem`));

	mixin(SMixInStructGetSet!(TREE_NODE)("parent", q{TvInsertItem.hParent}));
	
	///
	static enum : TREE_NODE {
		ROOT  = TVI_ROOT , /// アイテムをルートアイテムとして追加します。
		FIRST = TVI_FIRST, /// アイテムをリストの最初の位置に挿入します。
		LAST  = TVI_LAST , /// アイテムをリストの最後の位置に挿入します。
		SORT  = TVI_SORT , /// アイテムをアルファベット順にリストに挿入します。
	}
	mixin(SMixInStructGetSet!(TREE_NODE)("insertAfter", q{TvInsertItem.hInsertAfter}));


	TREEITEM* item() {
		return cast(TREEITEM*)&TvInsertItem.itemex;
	}
	
}

///
struct TREEHITTEST {
	TVHITTESTINFO TvHitTestInfo;
	mixin(SMixInStructHiddenOriginal!(TVHITTESTINFO)(`TvHitTestInfo`));

	ref POINT point() {
		return TvHitTestInfo.pt;
	}

	enum FLAGS {
		ABOVE           = TVHT_ABOVE           , /// Above the client area.
		BELOW           = TVHT_BELOW           , /// Below the client area.
		NOWHERE         = TVHT_NOWHERE         , /// In the client area, but below the last item.
		ONITEM          = TVHT_ONITEM          , /// On the bitmap or label associated with an item.
		ONITEMBUTTON    = TVHT_ONITEMBUTTON    , /// On the button associated with an item.
		ONITEMICON      = TVHT_ONITEMICON      , /// On the bitmap associated with an item.
		ONITEMINDENT    = TVHT_ONITEMINDENT    , /// In the indentation associated with an item.
		ONITEMLABEL     = TVHT_ONITEMLABEL     , /// On the label (string) associated with an item.
		ONITEMRIGHT     = TVHT_ONITEMRIGHT     , /// In the area to the right of an item.
		ONITEMSTATEICON = TVHT_ONITEMSTATEICON , /// On the state icon for a tree view item that is in a user-defined state.
		TOLEFT          = TVHT_TOLEFT          , /// To the left of the client area.
		TORIGHT         = TVHT_TORIGHT         , /// To the right of the client area.
	}
	mixin(SMixInStructGetSet!(FLAGS)("flags", q{TvHitTestInfo.flags}));
	mixin(SMixInStructGetSet!(TREE_NODE)("item", q{TvHitTestInfo.hItem}));
	
}

///
typedef UINT TREE_ACTION;

///
struct TREEVIEWMESSAGE {
	NMTREEVIEW NmTreeView;
	mixin(SMixInStructHiddenOriginal!(NMTREEVIEW)(`NmTreeView`));
	
	const NOTIFY* notify() {
		return cast(NOTIFY*)&NmTreeView.hdr;
	}

	enum ACTION: TREE_ACTION {
		UNKNOWN  = TVC_UNKNOWN,
		MOUSE    = TVC_BYMOUSE,
		KEYBOARD = TVC_BYKEYBOARD,
	}
	TREEITEM* itemOld() {
		return cast(TREEITEM*)&NmTreeView.itemOld;
	}
	TREEITEM* itemNew() {
		return cast(TREEITEM*)&NmTreeView.itemNew;
	}
	ref POINT dragPoint() {
		return NmTreeView.ptDrag;
	}
}


