﻿/**
アイコン拡張。

nemuxi.negui.draw.iconを使用してちょっとやれること増やしたよん。

History:
	1.031:
		[@] 過剰だったんでログ取りの廃止。
*/
module nemuxi.image.icon;

debug import std.stdio: wl = writefln, pl = printf;

import std.contracts;

import win32.windows;

import nemuxi.negui.system.base;
import nemuxi.negui.draw.icon;
import nemuxi.system.exception;
import nemuxi.system.staticdata;
import nemuxi.negui.file.file;
import nemuxi.system.log;

/**
アイコンを取得。

ABのGetIconHandleそのまま。

Params:
	Address = ファイルアドレス。

	IconIndex = アイコンインデックス。
	            Addressがアイコンを持っている(実行形式)の場合に
	            IconIndexがAddressより大きい場合は0に設定される。

	IconFixed = アイコンの大きさ。

Return:
	アイコン。

Exception:
	ミスった場合はNemuxiException、たんまにWin32 Exception。
	Addressが行方不明の場合もなげる。

In:
	Address.length > 0

History:
	1.00β11:
		DLLのアイコンを取得できていなかった不具合の修正。
		ファイルの有無で例外を投げる。
*/
Icon GetFileIcon(in Text Address, int IconIndex, Icon.FIXED IconFixed)
in {
	assert(Address.length);
}
body {
	enforce(FILE.isExistence(Address), new NemuxiException(Address ~ "がない"));
	SHFILEINFO FileInfo=void;

	const pAddress=Address.ptr;
	immutable dll=PATH.cmp(PATH.getExtension(Address), Text("dll"))==0;

	//ファイルは実行形式？
	bool IsBin=false;
	int IconCount;
	if(dll || SHGetFileInfo(pAddress, 0, &FileInfo, FileInfo.sizeof, SHGFI_EXETYPE)) {
		IsBin = true;
		//保持アイコン総数
		IconCount=ExtractIconEx(pAddress, -1, null, null, 0);
		if(!IconCount && dll) {
			// DLLだけどアイコンを保持していない。
			IsBin = false;
		}
	}
	if(IsBin) {
		// ファイルは実行形式
		if(IconIndex > IconCount) {
			// 指定アイコンインデックスが多すぎるので最小値へ
			IconIndex = 0;
		}

		with(Icon.FIXED) switch(IconFixed) {
			case SMALL:
				ExtractIconEx(pAddress, IconIndex, null, &FileInfo.hIcon, 1);
				break;
			case NORMAL:
				ExtractIconEx(pAddress, IconIndex, &FileInfo.hIcon, null, 1);
				break;
			default:
				assert(false);
		}
	} else {
		// ふつーのファイル
		int IconFlag=void;
		with(Icon.FIXED) switch(IconFixed) {
			case SMALL:
				IconFlag = SHGFI_SMALLICON;
				break;
			case NORMAL:
				IconFlag = SHGFI_LARGEICON;
				break;
			default:
				assert(false);
		}
		if(!SHGetFileInfo(pAddress, FILE_ATTRIBUTE_ARCHIVE, &FileInfo, FileInfo.sizeof, SHGFI_ICON | IconFlag)) {
			throw new NemuxiException("GetFileIcon:SHGetFileInfo(" ~ Address.toString() ~ ")");
		}
	}
	
	return new Icon(FileInfo.hIcon, true);
}

debug(icon) unittest {
	GetFileIcon(Text(`C:\Program Files\Mozilla Firefox\plugins`),0,Icon.FIXED.SMALL);
}
/**

Bugs:
	アイコン名と内容が多分違う。
*/
enum SYSICON: ushort {
	NEMUXI,
	URI,
	MULTI,
	TASKTRAY,
	
	FOLDER,
	FILE,

	ELEMENT_DELETE,
	ELEMENT_EDIT,
	ELEMENT_MOVE_DOWN,
	ELEMENT_MOVE_UP,
	GROUP_ITEM_ADD,
	GROUP_ITEM_INS,
	GROUP_MAIN_ADD,
	GROUP_MAIN_INS,
	GROUP_SUB_ADD,
	GROUP_SUB_INS,

}
/**
システムアイコンの取得。

当moduleの「システムアイコン」は

①Windows内部アイコン<br />
②ﾈﾑぃのリソースアイコン<br />
③②が無効な場合にﾈﾑぃの内部リソースアイコン

を指す。

Params:
	SysIcon = アイコンフラグ。

	IconFixed = アイコンサイズ。

Return:
	アイコン。

Exception:
	Windows内部アイコンの取得に失敗した場合にNemuxiExceptionを投げる。

History:
	1.030:
		[P] 処理内容変更。

	1.022:
		[S] final switchの導入。
*/
Icon GetSystemIcon(SYSICON SysIcon, Icon.FIXED IconFixed) {
	immutable size=Icon.fixedToSize(IconFixed);
	
	Text IconName=void;
	final switch(SysIcon) {
		case SYSICON.NEMUXI:    IconName = "nemuxi.ico";            break;
		case SYSICON.URI:       IconName = "uri.ico";               break;
		case SYSICON.MULTI:     IconName = "multi.ico";             break;
		case SYSICON.TASKTRAY:  IconName = "tasktray.ico";          break;
		
		case SYSICON.FOLDER:    IconName = "folder.ico";            break;
		case SYSICON.FILE:      IconName = "file.ico";              break;

		case SYSICON.ELEMENT_DELETE:    IconName = "element_delete.ico";    break;
		case SYSICON.ELEMENT_EDIT:      IconName = "element_edit.ico";      break;
		case SYSICON.ELEMENT_MOVE_DOWN: IconName = "element_move_down.ico"; break;
		case SYSICON.ELEMENT_MOVE_UP:   IconName = "element_move_up.ico";   break;
		case SYSICON.GROUP_ITEM_ADD:    IconName = "group_item_add.ico";    break;
		case SYSICON.GROUP_ITEM_INS:    IconName = "group_item_ins.ico";    break;
		case SYSICON.GROUP_MAIN_ADD:    IconName = "group_main_add.ico";    break;
		case SYSICON.GROUP_MAIN_INS:    IconName = "group_main_ins.ico";    break;
		case SYSICON.GROUP_SUB_ADD:     IconName = "group_sub_add.ico";     break;
		case SYSICON.GROUP_SUB_INS:     IconName = "group_sub_ins.ico";     break;
	}

	Text IconAddress=PATH.addFolderSep(StaticData.resourceFolder) ~ IconName;
	if(FILE.isExistence(IconAddress)) {
		return GetFileIcon(IconAddress, 0, IconFixed);
	} else {
		return new Icon(LoadImage(
			GetModuleHandle(null),
			GetIconResource(SysIcon),
			IMAGE_ICON,
			size, size,
			LR_SHARD
		), false);
	}
	
}
/**
History:
	1.022:
		[S] final switchの導入。
*/
private wchar* GetIconResource(SYSICON SysIcon) {
	int IconName;
	final switch(SysIcon) {
		case SYSICON.NEMUXI:   IconName = 100; break;
		case SYSICON.URI:      IconName = 106; break;
		case SYSICON.MULTI:    IconName = 108; break;
		case SYSICON.TASKTRAY: IconName = 105; break;
		
		case SYSICON.FOLDER:   IconName = 110; break;
		case SYSICON.FILE:     IconName = 112; break;

		case SYSICON.ELEMENT_DELETE:    IconName = 210; break;
		case SYSICON.ELEMENT_EDIT:      IconName = 211; break;
		case SYSICON.ELEMENT_MOVE_DOWN: IconName = 212; break;
		case SYSICON.ELEMENT_MOVE_UP:   IconName = 213; break;
		case SYSICON.GROUP_ITEM_ADD:    IconName = 214; break;
		case SYSICON.GROUP_ITEM_INS:    IconName = 215; break;
		case SYSICON.GROUP_MAIN_ADD:    IconName = 216; break;
		case SYSICON.GROUP_MAIN_INS:    IconName = 217; break;
		case SYSICON.GROUP_SUB_ADD:     IconName = 218; break;
		case SYSICON.GROUP_SUB_INS:     IconName = 219; break;
	}
	return cast(wchar*)IconName;
}
/+
#define IDI_ICON1                               100
#define IDI_TASKTRAY                            105
#define IDI_URI                                 106
#define IDI_MULTI                               108

#define IDI_FOLDER                              110
#define IDI_FILE                                112

#define IDI_ELEMENT_DELETE                      210
#define IDI_ELEMENT_EDIT                        211
#define IDI_ELEMENT_MOVE_DOWN                   212
#define IDI_ELEMENT_MOVE_UP                     213
#define IDI_GROUP_ITEM_ADD                      214
#define IDI_GROUP_ITEM_INS                      215
#define IDI_GROUP_MAIN_ADD                      216
#define IDI_GROUP_MAIN_INS                      217
#define IDI_GROUP_SUB_ADD                       218
#define IDI_GROUP_SUB_INS                       219

+/































