﻿/**

*/
module nemuxi.file.exec;

debug import std.stdio: wl = writefln, pl = printf;

debug(exec) void main() {}

import std.string;
import std.contracts;
import core.thread;

import win32.windows;


//import nemuxi.base;
import nemuxi.negui.system.base;
import nemuxi.negui.system.type.enumerated;
import nemuxi.negui.file.file;
import nemuxi.negui.negui;
import nemuxi.system.exception;
//import nemuxi.negui.system.type;
//import nemuxi.negui.system.error;

///
class ExecuteException: NemuxiException {
	mixin MixInNemuxiException;
}


/// フォルダを開く方法。
enum FOLDER {
	NORMAL,
	EXPLORER,
	PROGRAM,
}

/**
ファイル実行。

Params:
	FileAddress = 実行するファイル。
	
	Option = FileAddressに渡すオプション。

	WorkFolder = 作業フォルダ。
	             無効なものであればFileAddressの親フォルダを使用。

	Show = 実行時の表示フラグ。

	OpenExplorer = FileAddressがフォルダの場合にtrueならエクスプローラで、falseなら普通に開く。
	
	hWnd = 実行ファイルの親ウィンドウ。

Retrun:

Exception:
	FileAddressが存在しなければNemuxiExceptionを投げる。

History:
	1.00β16:
		[P] 例外の種類を変更。
		
	1.00β11:
		フォルダ判定部分がファイル存在判定になっていた部分を修正。
*/
HINSTANCE FileExecute(
	in Text FileAddress,
	in Text Option,
	in Text WorkFolder,
	in SHOW Show,
	in bool OpenExplorer=false,
	in NeGui gui=null
)
in {
	assert(FileAddress.length());
//	assert(Show != SHOW.NONE);
}
body {
	SHELLEXECUTEINFO Shell;
	
	try {
		Shell.cbSize = Shell.sizeof;
		Shell.hwnd   = gui ? gui(): GetDesktopWindow();
		Shell.fMask  = SEE_MASK_DOENVSUBST | SEE_MASK_FLAG_NO_UI;
		if(!FILE.isExistence(FileAddress)) {
			auto Spliter=FileAddress.find(' ');
			Text OptionEx;
			if(Spliter != -1) {
				OptionEx = FileAddress[Spliter+1..FileAddress.length];
			} else {
				Spliter = FileAddress.length;
			}
			// path
			auto ret = ShellExecute(
				Shell.hwnd,
				null,
				FileAddress[0..Spliter].ptr,
				(OptionEx ~ Option).ptr,
				null,
				Show
			);
			if(ret > cast(HANDLE)32) {
				return ret;
			} else {
				ERR.code = cast(uint)ret;
				throw new ExecuteException(
					format("%s[%s]", ERR.toString(), FileAddress),
					EC.FILE_EXECUTE_PATH
				);
			}
		} else if(FILE.isFile(FileAddress)) {
			// ファイルの場合
			bool Wark=void;
			if(WorkFolder.length) {
				try {
					Wark = FILE.isFolder(WorkFolder);
				} catch(NeGuiFileException e) {
					Wark = false;
				}
			} else {
				Wark = false;
			}
			Shell.lpFile       = FileAddress.ptr;
			Shell.lpParameters = Option.ptr;
			Shell.lpDirectory  = Wark ? WorkFolder.ptr: PATH.ownerFolder(FileAddress).ptr,
			Shell.nShow        = Show;
			Shell.lpVerb       = Text("open").ptr;
		} else {
			// フォルダの場合
			Shell.lpFile = FileAddress.ptr;
			Shell.lpVerb = OpenExplorer ? Text("explore").ptr: Text("open").ptr;
			Shell.nShow  = Show;
		}
		ERR.initialize;
		enforce(
			ShellExecuteEx(&Shell),
			new ExecuteException(format("%s [%s]", ERR.toString(), FileAddress.toString), EC.FILE_EXECUTE_PROPERTY)
		);
	} catch(Exception e) {
		throw new ExecuteException("ファイル実行例外", EC.FILE, e);
	}

	return Shell.hInstApp;
}
debug(exec) unittest {
	//FileExecute(Text(`C:\Program Files\Opera\Opera.exe`), Text(`http://www.usamimi.info/~wokonomono/`), Text.emptyText, NeGui.SW.SHOW);
	FileExecute(Text(`control`), Text(`mouse`), Text.emptyText, SHOW.SHOWNORMAL);
}

/**
フォルダオープン。

内部でFileExecuteを呼んでいる。

Params:
	FolderAddress = フォルダのアドレス。
	                x\y\z\って感じで最後が\のものがﾈﾑぃではフォルダ認定される。

	OpenState = どのようにしてフォルダを開くか。

	Show = 実行時の表示フラグ。

	OpenProgram = OpenStateがFOLDER.PROGRAMの時にフォルダを開くプログラムを指定。
	              FOLDER.PROGRAMの場合はnullだと表明違反。
	              こいつの後ろにFolderAddressがそのまんま引っ付くのでこれを呼ぶまでに加工すべし。
*/
HINSTANCE FolderExecute(
	in Text FolderAddress,
	in FOLDER OpenState,
	in SHOW Show,
	in Text* OpenProgram=null
)
in {
	assert(FolderAddress.length);
	if(OpenState == FOLDER.PROGRAM)
		assert(OpenProgram, OpenProgram.toString);
}
body {
	try {
		if(OpenState != FOLDER.PROGRAM) {
			Text SafeFolderAddress;
			if(FILE.isExistence(FolderAddress)) {
				SafeFolderAddress = PATH.addFolderSep(FolderAddress);
			} else {
				SafeFolderAddress = FolderAddress;
			}
			return FileExecute(
				SafeFolderAddress,
				Text.emptyText(),
				Text.emptyText(),
				Show,
				OpenState == FOLDER.EXPLORER
			);
		} else {
			enforce(
				FolderAddress.length,
				new NeGuiFileException(Text("肝心のフォルダパスが完全に無効"))
			);
		}
		return FileExecute(
			*OpenProgram,
			FolderAddress,
			FolderAddress,
			Show
		);
	} catch(Exception e) {
		throw new ExecuteException("フォルダ実行, " ~ FolderAddress.toString, EC.FILE, e);
	}
}
debug(exec) unittest {
	FolderExecute(
		Text(`::{20D04FE0-3AEA-1069-A2D8-08002B30309D}\::{21EC2020-3AEA-1069-A2DD-08002B30309D}\::{7007ACC7-3202-11D1-AAD2-00805FC1270E}`),
		FOLDER.NORMAL,
		SHOW.SHOWNORMAL
	);
}
/**
ABから移行、例外追加。
*/
HINSTANCE PropertyExecute(in Text FileAddress, in NeGui gui=null) {
	SHELLEXECUTEINFO Shell;

	with(Shell) {
		cbSize = Shell.sizeof;
		hwnd   = gui ? gui(): GetDesktopWindow();
		nShow  = SW_SHOWNORMAL;
		fMask  = SEE_MASK_NOCLOSEPROCESS | SEE_MASK_INVOKEIDLIST | SEE_MASK_FLAG_NO_UI;
		lpVerb = Text("properties").ptr;
		lpFile = FileAddress.ptr;
	}
	ERR.initialize;
	enforce(
		ShellExecuteEx(&Shell),
		new NeGuiFileException(Text("%s[%s]", ERR.toText, FileAddress))
	);

	return Shell.hInstApp;
}

/**
<del>Windowsじゃstd.process.shell使えないっぽいのでそれっぽいのを実装。</del>

Exception:
	実行したときに非常に変だったらNemuxiExceptionを投げる。

History:
	1.00β15:
		別に更新してないけど2.032でstd.process.shellが使える…どうしたもんか。
*/
class ShellExec: Thread {
	private {
		Text CommandLine;
		DWORD ReturnCode;
	}
	DWORD returnCode() {
		return ReturnCode;
	}
	Text ShellCode;

	this(in Text CommandLine) {
		this.CommandLine = CommandLine;
		super(&run);
	}

	private void run() {
		SECURITY_ATTRIBUTES SecAtributes=void;
		with(SecAtributes) {
			nLength = SecAtributes.sizeof;
			lpSecurityDescriptor = null;
			bInheritHandle = true;
		}

		HANDLE hRead, hWrite;
		
		enforce(CreatePipe(&hRead, &hWrite, &SecAtributes, 0), new NemuxiException(ERR.toString(), EC.PROCCESS));

		STARTUPINFO StartUpInfo;
		with(StartUpInfo) {
			cb          = StartUpInfo.sizeof;
			dwFlags     = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
			wShowWindow = SW_SHOWDEFAULT;
			hStdInput   = hRead;
			hStdOutput  = hWrite;
			hStdError   = hWrite;
		}
		PROCESS_INFORMATION ProccessInfo;
		enforce(
			// 引数多いな
			CreateProcess(
				null,            // 実行可能モジュールの名前
				CommandLine.ptr, // コマンドラインの文字列
				null,            // セキュリティ記述子
				null,            // セキュリティ記述子
				true,            // ハンドルの継承オプション
				0,               // 作成のフラグ
				null,            // 新しい環境ブロック
				null,            // カレントディレクトリの名前
				&StartUpInfo,    // スタートアップ情報
				&ProccessInfo    // プロセス情報
			),
			new NemuxiException(CommandLine.toString() ~ ":" ~ ERR.toString(), EC.PROCCESS_CREATE)
		);
		// 落ち着くまで待機
		WaitForInputIdle(ProccessInfo.hProcess, INFINITE);
		CloseHandle(ProccessInfo.hThread);

		// 終了まで待機。
		WaitForSingleObject(ProccessInfo.hProcess, INFINITE);

		// 終了コード取得
		GetExitCodeProcess(ProccessInfo.hProcess, &ReturnCode);
		CloseHandle(ProccessInfo.hProcess);

		size_t NoPipeRead;
		do{
			size_t PipeRead;
			NoPipeRead=0;

			enforce(PeekNamedPipe(hRead, null, 0, null, &PipeRead, &NoPipeRead), new NemuxiException(ERR.toString(), EC.PROCCESS));
			if(PipeRead) {
				size_t FileRead;
				auto ReadData=new char[PipeRead];
				enforce(ReadFile(hRead, ReadData.ptr, PipeRead, &FileRead, NULL), new NemuxiException(ERR.toString(), EC.PROCCESS));
				if(FileRead) {
					ShellCode ~= cast(string)ReadData[0..FileRead];
				} else {
					break;
				}
			}
		} while(NoPipeRead);
	}

	override string toString() {
		return format("%s%s[CODE = %s]", ShellCode.toString, newline, ReturnCode);
	}
}
debug(exec) unittest {
	auto n=new ShellExec(Text("ping 192.168.1.1"));
	n.start;n.join;
	wl("%s",n.toString);
}
